/* PragmaDev RTDS Nucleus integration */
#ifndef _RTDS_MACRO_H_
#define _RTDS_MACRO_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "stdio.h"
#include "stdlib.h"

/* RTDS include */
#include "RTDS_OS.h"
#include "RTDS_Error.h"
#ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
#include "RTDS_BackTrace.h"
#endif

/* ************************************************************************************ *
 * TASK ENTRY POINT DEFINITION AND PROTOTYPE
 * ************************************************************************************ */

#define RTDS_TASK_ENTRY_POINT( TASK_NAME ) \
    VOID TASK_NAME( UNSIGNED option, VOID * pointer )

#define RTDS_TASK_ENTRY_POINT_PROTO( TASK_NAME ) \
    extern RTDS_TASK_ENTRY_POINT( TASK_NAME )

/* ************************************************************************************ *
 * SDL KEYWORD DEFINITION
 * Defines the SDL PARENT keyword. parentQueueId is given when taskSpawn is done.
 * ************************************************************************************ */
#define PARENT    RTDS_currentContext->parentSdlInstanceId
#define SELF      RTDS_currentContext->mySdlInstanceId
#define OFFSPRING RTDS_currentContext->offspringSdlInstanceId
#define SENDER    RTDS_senderId

/* ************************************************************************************ *
 * SYSTEM MACROS
 * ************************************************************************************ */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_BACK_TRACE_MAX_EVENT_NUM )
    #ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
        #define RTDS_SIMULATOR_TRACE(EVENT, PARAM1, PARAM2, PARAM3) \
            RTDS_globalTraceEntry.event = EVENT; \
            RTDS_globalTraceEntry.eventParameter1 = ( void * )PARAM1; \
            RTDS_globalTraceEntry.eventParameter2 = ( long )PARAM2; \
            RTDS_globalTraceEntry.currentContext = (RTDS_GlobalProcessInfo *) PARAM3; \
            RTDS_TraceAdd(); \
            RTDS_globalSystemTime = RTDS_GetSystemTime(); \
            RTDS_DummyTraceFunction()
    #else
        #define RTDS_SIMULATOR_TRACE(EVENT, PARAM1, PARAM2, PARAM3) \
            RTDS_globalTraceEntry.event = EVENT; \
            RTDS_globalTraceEntry.eventParameter1 = ( void * )PARAM1; \
            RTDS_globalTraceEntry.eventParameter2 = ( long )PARAM2; \
            RTDS_globalTraceEntry.currentContext = ( RTDS_GlobalProcessInfo * ) PARAM3; \
            RTDS_globalSystemTime = RTDS_GetSystemTime(); \
            RTDS_DummyTraceFunction()
    #endif
#else
     #define RTDS_SIMULATOR_TRACE(EVENT, PARAM1, PARAM2, PARAM3)
#endif

/*
 * MACRO RTDS_SYSTEM_ERROR:
 * ------------------------
 * Macro called when a system error is detected in RTDS 'kernel'
 * Needs to be further defined by user.
 */
#ifndef RTDS_SYSTEM_ERROR
    #define RTDS_SYSTEM_ERROR( errorNumber ) \
        { \
        RTDS_SIMULATOR_TRACE( RTDS_systemError, errorNumber, NULL, NULL ); \
        printf( "RTDS error 0x%X\nCheck RTDS_Error.h file for explanations.\n", errorNumber ); \
        exit( errorNumber ); \
        }
#endif

/* ************************************************************************************ *
 * MEMORY MACROS
 * ************************************************************************************ */

/*
 * MACRO RTDS_MALLOC:
 * ---------------------
 * Memory allocation
 */
#define RTDS_MALLOC( SIZE ) \
    RTDS_Malloc( SIZE )

/*
 * MACRO RTDS_FREE:
 * -------------------
 * Memory liberation
 */
#define RTDS_FREE( PTR ) \
    RTDS_Free( ( VOID * )( PTR ) );

/*
 * MACRO RTDS_MEMCPY:
 * ---------------------
 * Memory copy
 */
#define RTDS_MEMCPY( DEST, SRC, SIZE ) \
    memcpy( DEST, SRC, SIZE )

/* ************************************************************************************ *
 * TIMER MACROS
 * ************************************************************************************ */

/*
 * MACRO RTDS_RESET_TIMER:
 * --------------------------
 * Cancels a timer
 */
#define RTDS_RESET_TIMER( TIMER_NUMBER ) \
    RTDS_StopTimer( TIMER_NUMBER, &( RTDS_currentContext->timerList ), RTDS_currentContext )

/*
 * MACRO RTDS_SET_TIMER:
 * ------------------------
 * Initializes and starts a timer
 */
#define RTDS_SET_TIMER( TIMER_NUMBER, DELAY ) \
    RTDS_RESET_TIMER( TIMER_NUMBER ); \
    RTDS_StartTimer( SELF, TIMER_NUMBER, RTDS_GetTimerUniqueId( RTDS_currentContext->timerList ), DELAY, &( RTDS_currentContext->timerList ), RTDS_currentContext )

/* ************************************************************************************ *
 * MESSAGE MACROS
 * ************************************************************************************ */

/*
 * MACRO RTDS_MSG_QUEUE_READ:
 * -----------------------------
 * Reads the next message in the process's message queue
 * Memory needs to be allocated before calling the macro
 */
#define RTDS_MSG_QUEUE_READ( CURRENT_MESSAGE ) \
    RTDS_MsgReceive( SELF, &( CURRENT_MESSAGE ) )

/*
 * MACRO RTDS_MSG_QUEUE_SEND_TO_ID:
 * ----------------------------------
 * Sends a message to a process's message queue using its queue id.
 */
#ifndef RTDS_MSG_QUEUE_SEND_TO_ID
#define RTDS_MSG_QUEUE_SEND_TO_ID( MESSAGE_NUMBER, LENGTH_DATA, P_DATA, RECEIVER_ID ) \
    RTDS_MsgQueueSend( MESSAGE_NUMBER, LENGTH_DATA, ( unsigned char * ) P_DATA, RECEIVER_ID, SELF, RTDS_currentContext )
#endif

/*
 * MACRO RTDS_MSG_QUEUE_SEND_TO_ENV:
 * ------------------------------------
 * Sends a message to the environment.
 */
#ifndef RTDS_MSG_QUEUE_SEND_TO_ENV
#define RTDS_MSG_QUEUE_SEND_TO_ENV( MESSAGE_NUMBER, LENGTH_DATA, P_DATA ) \
    RTDS_MsgQueueSend( MESSAGE_NUMBER, LENGTH_DATA, ( unsigned char * ) P_DATA, RTDS_GetProcessInstanceId( RTDS_process_RTDS_Env ), SELF, RTDS_currentContext )
#endif

/*
 * MACRO RTDS_MSG_QUEUE_SEND_TO_NAME:
 * ------------------------------------
 * Sends a message to a process's message queue using its name.
 */
#ifndef RTDS_MSG_QUEUE_SEND_TO_NAME
#define RTDS_MSG_QUEUE_SEND_TO_NAME( MESSAGE_NUMBER, LENGTH_DATA, P_DATA, RECEIVER_NAME, RECEIVER_NUMBER ) \
    RTDS_MsgQueueSend( MESSAGE_NUMBER, LENGTH_DATA, ( unsigned char * ) P_DATA, RTDS_GetProcessInstanceId( RECEIVER_NUMBER ), SELF, RTDS_currentContext )
#endif

/*
 * MACRO RTDS_MSG_SAVE:
 * -----------------------
 * Saves a message to the save queue chained list
 */
#define RTDS_MSG_SAVE( CURRENT_MESSAGE ) \
    RTDS_SIMULATOR_TRACE( RTDS_messageSaved, CURRENT_MESSAGE, SELF, RTDS_currentContext ); \
    CURRENT_MESSAGE->next = NULL; \
    if ( RTDS_currentContext->writeSaveQueue == NULL ) \
        { \
        RTDS_currentContext->writeSaveQueue = CURRENT_MESSAGE; \
        } \
    else \
        { \
        RTDS_MessageHeader * RTDS_MSG_SAVE_message; \
        for( RTDS_MSG_SAVE_message = RTDS_currentContext->writeSaveQueue; RTDS_MSG_SAVE_message->next != NULL; RTDS_MSG_SAVE_message = RTDS_MSG_SAVE_message->next ); \
        RTDS_MSG_SAVE_message->next = CURRENT_MESSAGE; \
        } \
    CURRENT_MESSAGE = NULL

/*
 * MACRO RTDS_MSG_INPUT_ERROR:
 * --------------------------
 * Macro called in the generated code if the pointer on the received parameters is NULL.
 */
#define RTDS_MSG_INPUT_ERROR \
    RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_INPUT )

/* ************************************************************************************ *
 * SEMAPHORE MACROS
 * ************************************************************************************ */

/*
 * MACRO RTDS_BINARY_SEMAPHORE_CREATE:
 * -------------------------------------
 * Creates a binary semaphore
 */
#define RTDS_BINARY_SEMAPHORE_CREATE( SEMAPHORE_NAME, SEMAPHORE_NUMBER, OPTIONS, INITIAL_STATE ) \
    RTDS_Sem_Info_Insert( SEMAPHORE_NUMBER, RTDS_Semaphore_Create( SEMAPHORE_NAME, OPTIONS, INITIAL_STATE ), RTDS_currentContext )

/*
 * MACRO RTDS_MUTEX_SEMAPHORE_CREATE:
 * -------------------------------------
 * Creates a mutex semaphore
 */
#define RTDS_MUTEX_SEMAPHORE_CREATE( SEMAPHORE_NAME, SEMAPHORE_NUMBER, OPTIONS ) \
    RTDS_SYSTEM_ERROR( RTDS_ERROR_CREATE_MUTEX_NOT_APPLICABLE )

/*
 * MACRO RTDS_COUNTING_SEMAPHORE_CREATE:
 * ---------------------------------------
 * Creates a counting semaphore
 */
#define RTDS_COUNTING_SEMAPHORE_CREATE( SEMAPHORE_NAME, SEMAPHORE_NUMBER, OPTIONS, INITIAL_COUNT ) \
    RTDS_Sem_Info_Insert( SEMAPHORE_NUMBER, RTDS_Semaphore_Create( SEMAPHORE_NAME, OPTIONS, INITIAL_COUNT ), RTDS_currentContext )

/*
 * MACRO RTDS_SEMAPHORE_DELETE:
 * -----------------------------
 * Deleting a semaphore by id
 */
#define RTDS_SEMAPHORE_DELETE( SEMAPHORE_ID ) \
    RTDS_Sem_Delete( SEMAPHORE_ID, RTDS_currentContext )

/*
 * MACRO RTDS_SEMAPHORE_DELETE:
 * -----------------------------
 * Deleting a semaphore by name
 */
#define RTDS_SEMAPHORE_NAME_DELETE( SEMAPHORE_NAME, SEMAPHORE_NUMBER ) \
    RTDS_Sem_Delete( RTDS_GetSemaphoreId( SEMAPHORE_NUMBER ), RTDS_currentContext )

/*
 * MACRO RTDS_SEMAPHORE_TAKE:
 * -----------------------------
 * Takes a semaphore by id
 */
#define RTDS_SEMAPHORE_ID_TAKE( SEMAPHORE_ID, TIME_OUT) \
    RTDS_SemaphoreIdTake( SEMAPHORE_ID, TIME_OUT, RTDS_currentContext )

/*
 * MACRO RTDS_SEMAPHORE_TAKE:
 * -----------------------------
 * Takes a semaphore by name
 */
#define RTDS_SEMAPHORE_NAME_TAKE( SEMAPHORE_NAME, SEMAPHORE_NUMBER, TIME_OUT ) \
    RTDS_SemaphoreIdTake( RTDS_GetSemaphoreId( SEMAPHORE_NUMBER ), TIME_OUT, RTDS_currentContext )

/*
 * MACRO RTDS_SEMAPHORE_GIVE:
 * -----------------------------
 * Gives a semaphore by id
 */
#define RTDS_SEMAPHORE_ID_GIVE( SEMAPHORE_ID ) \
    RTDS_SIMULATOR_TRACE( RTDS_semGive, SEMAPHORE_ID, NULL, RTDS_currentContext ); \
    RTDS_SemaphoreIdGive( SEMAPHORE_ID )

/*
 * MACRO RTDS_SEMAPHORE_GIVE:
 * -----------------------------
 * Gives a semaphore by name
 */
#define RTDS_SEMAPHORE_NAME_GIVE( SEMAPHORE_NAME, SEMAPHORE_NUMBER ) \
    RTDS_SIMULATOR_TRACE( RTDS_semGive, RTDS_GetSemaphoreId( SEMAPHORE_NUMBER ), NULL, RTDS_currentContext ); \
    RTDS_SemaphoreIdGive( RTDS_GetSemaphoreId( SEMAPHORE_NUMBER ) )

/* ************************************************************************************ *
 * PROCESS MACROS
 * ************************************************************************************ */

/*
 * MACRO RTDS_PROCESS_CREATE:
 * -----------------------------
 * Creates a new task representing an SDL process
 */
#define RTDS_PROCESS_CREATE( PROCESS_NAME, PROCESS_NUMBER, FUNCTION_ADDRESS, PRIORITY ) \
    RTDS_ProcessCreate( PROCESS_NAME, PROCESS_NUMBER, FUNCTION_ADDRESS, PRIORITY, RTDS_HOLD, RTDS_currentContext )

/*
 * MACRO RTDS_STARTUP_PROCESS_CREATE:
 * -----------------------------
 * Used by Startup task to create a new task representing an SDL process
 */
#define RTDS_STARTUP_PROCESS_CREATE( PROCESS_NAME, PROCESS_NUMBER, FUNCTION_ADDRESS, PRIORITY ) \
    RTDS_ProcessCreate( PROCESS_NAME, PROCESS_NUMBER, FUNCTION_ADDRESS, PRIORITY, RTDS_NO_HOLD, RTDS_currentContext )

/*
 * MACRO RTDS_PROCESS_KILL:
 * --------------------------
 * Kills a task representing an SDL process. An SDL task can only kill itself.
 */
#define RTDS_PROCESS_KILL \
    RTDS_ProcessKill( RTDS_currentContext ); \
    return

/* ************************************************************************************ *
 * FINITE STATE MACHINE
 * ************************************************************************************ */
/*
 * MACRO RTDS_PROCEDURE_CLEAN_UP:
 * ------------------------------
 * Macro called at the end of a procedure to free last message and re-organize
 * save queue if procedure contained states.
 */
#define RTDS_PROCEDURE_CLEAN_UP \
    RTDS_TransitionCleanUp( RTDS_currentContext, 0 )

/*
 * MACRO RTDS_SDL_STATE_SET:
 * -----------------------------
 * Sets the SDL state of the current process
 */
#define RTDS_SDL_STATE_SET( NEW_SDL_STATE ) \
    RTDS_currentContext->sdlState = NEW_SDL_STATE; \
    RTDS_SIMULATOR_TRACE( RTDS_sdlStateSet, RTDS_currentContext->mySdlInstanceId, NEW_SDL_STATE, RTDS_currentContext )

#ifdef __cplusplus
}
#endif

#endif /* _RTDS_MACRO_H_ */

