/* PragmaDev RTDS uITRON 4 integration */
/* ****************************************************************
 * RTDS UITRON MACROS DECLARATION
 * ---------------------------------------------------------------
 *
 ****************************************************************** */
#ifndef _RTDS_MACRO_H_
#define _RTDS_MACRO_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdlib.h>
#include "RTDS_OS.h"

#ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
#include "RTDS_BackTrace.h"
#endif

/*
 * SDL KEYWORD DEFINITION:
 * -------------------
 * Defines the SDL PARENT keyword. parentId is given when taskSpawn is done.
 */
#define PARENT          RTDS_currentContext->parentSdlInstanceId
#define SELF            RTDS_currentContext->mySdlInstanceId
#define OFFSPRING       RTDS_currentContext->offspringSdlInstanceId
#define SENDER          RTDS_senderId

/* ************************************************************************************ *
 * TASK ENTRY POINT DEFINITION
 ************************************************************************************* */
#define RTDS_TASK_ENTRY_POINT( TASK_NAME ) \
    void TASK_NAME( RTDS_GlobalProcessInfo * RTDS_currentContext )

#define RTDS_TASK_ENTRY_POINT_PROTO( TASK_NAME ) \
    extern RTDS_TASK_ENTRY_POINT( TASK_NAME )

/* ************************************************************************************ *
 * MEMORY MACROS
 * ************************************************************************************ */

/*
 * MACRO RTDS_MALLOC:
 * ---------------------
 * Memory allocation
 */
#define RTDS_MALLOC( SIZE ) \
    RTDS_Malloc( SIZE );

/*
 * MACRO RTDS_FREE:
 * -------------------
 * Memory liberation
 */
#define RTDS_FREE( PTR ) \
    RTDS_Free( RTDS_mempoll_id, PTR ); /* Parameter: ID mplid,VP PTR */

/*
 * MACRO RTDS_MEM_COPY:
 * ------------------------
 * Copy SIZE bytes from
 * SRC memory ptr to DEST
 */
#define RTDS_MEMCPY( DEST, SRC, SIZE ) \
    memcpy( DEST, SRC, SIZE )

/* ************************************************************************************ *
 * TIMER MACROS
 * ************************************************************************************ */

/*
 * MACRO RTDS_RESET_TIMER:
 * --------------------------
 * Cancels a timer
 */
#define RTDS_RESET_TIMER( TIMER_NUMBER ) \
    RTDS_StopTimer( TIMER_NUMBER, &( RTDS_currentContext->timerList ), RTDS_currentContext );

/*
 * MACRO RTDS_SET_TIMER:
 * ------------------------
 * Initializes and starts a timer
 */
#define RTDS_SET_TIMER( TIMER_NUMBER, DELAY ) RTDS_RESET_TIMER( TIMER_NUMBER ); \
    RTDS_StartTimer( SELF, TIMER_NUMBER, RTDS_GetTimerUniqueId( RTDS_currentContext->timerList ),DELAY, &( RTDS_currentContext->timerList ), RTDS_currentContext );

/* ************************************************************************************ *
 * MESSAGE MACROS
 * ************************************************************************************ */

/*
 * MACRO RTDS_MSG_QUEUE_READ:
 * -----------------------------
 * Reads the next message in the process's message queue
 * Memory needs to be allocated before calling the macro
 */
#define RTDS_MSG_QUEUE_READ( CURRENT_MESSAGE ) \
    RTDS_MsgReceive( SELF, CURRENT_MESSAGE );

/*
 * MACRO RTDS_MSG_QUEUE_SEND_TO_ID:
 * ----------------------------------
 * Sends a message to a process's message queue using its queue id.
 */
#ifndef RTDS_MSG_QUEUE_SEND_TO_ID
#define RTDS_MSG_QUEUE_SEND_TO_ID( MESSAGE_NUMBER, LENGTH_DATA, P_DATA, RECEIVER_ID ) \
    RTDS_MsgSend( MESSAGE_NUMBER, LENGTH_DATA, ( unsigned char * )P_DATA, RECEIVER_ID, SELF, RTDS_currentContext );
#endif

/*
 * MACRO RTDS_MSG_QUEUE_SEND_TO_ENV:
 * ------------------------------------
 * Sends a message to the environment.
 */
#ifndef RTDS_MSG_QUEUE_SEND_TO_ENV
#define RTDS_MSG_QUEUE_SEND_TO_ENV( MESSAGE_NUMBER, LENGTH_DATA, P_DATA ) \
    RTDS_MsgSend( MESSAGE_NUMBER, LENGTH_DATA, ( unsigned char * )P_DATA, RTDS_GetProcessInstanceId( RTDS_process_RTDS_Env ), SELF, RTDS_currentContext );
#endif

/*
 * MACRO RTDS_MSG_QUEUE_SEND_TO_NAME:
 * ------------------------------------
 * Sends a message to a process's message queue using its name.
 */
#ifndef RTDS_MSG_QUEUE_SEND_TO_NAME
#define RTDS_MSG_QUEUE_SEND_TO_NAME( MESSAGE_NUMBER, LENGTH_DATA, P_DATA, RECEIVER_NAME, RECEIVER_NUMBER ) \
    RTDS_MsgSend( MESSAGE_NUMBER, LENGTH_DATA, ( unsigned char * )P_DATA, RTDS_GetProcessInstanceId( RECEIVER_NUMBER ), SELF, RTDS_currentContext );
#endif

/*
 * MACRO RTDS_MSG_SAVE:
 * -----------------------
 * Saves a message to the save queue chained list
 */
#define RTDS_MSG_SAVE( CURRENT_MESSAGE ) \
    CURRENT_MESSAGE->next = NULL; \
    if( RTDS_currentContext->writeSaveQueue == NULL ) \
        { \
        RTDS_currentContext->writeSaveQueue = CURRENT_MESSAGE; \
        } \
    else \
        { \
        RTDS_MessageHeader * RTDS_MSG_SAVE_message; \
        for( RTDS_MSG_SAVE_message = RTDS_currentContext->writeSaveQueue; RTDS_MSG_SAVE_message->next != NULL; RTDS_MSG_SAVE_message = RTDS_MSG_SAVE_message->next ); \
        RTDS_MSG_SAVE_message->next = CURRENT_MESSAGE; \
        } \
        RTDS_SIMULATOR_TRACE( RTDS_messageSaved, CURRENT_MESSAGE, SELF, RTDS_currentContext ); \
        CURRENT_MESSAGE = NULL;

/*
 * MACRO RTDS_MSG_INPUT_ERROR:
 * --------------------------
 * Macro called in the generated code if the pointer on the received parameters is NULL.
 */
#define RTDS_MSG_INPUT_ERROR \
    RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_INPUT );

/* ************************************************************************************ *
 * SEMAPHORE MACROS
 * ************************************************************************************ */

/* Parameters are written by RTDS_code generator
 * and should match the RT_OS restriction
 */
#define RTDS_SEMAPHORE_TIME_OUT_FOREVER \
    RTDS_UITRON_SEM_TIMEOUT_FOREVER

/*
 * MACRO RTDS_BINARY_SEMAPHORE_CREATE:
 * -------------------------------------
 * Creates a binary semaphore, call function in RTDS_WINDOWS.c)
 */
#define RTDS_BINARY_SEMAPHORE_CREATE( SEMAPHORE_NAME, SEMAPHORE_NUMBER, OPTIONS, INITIAL_STATE ) \
    RTDS_Sem_Info_Insert( SEMAPHORE_NUMBER, RTDS_SemBCreate( OPTIONS, INITIAL_STATE ), RTDS_currentContext );
/*
 * MACRO RTDS_MUTEX_SEMAPHORE_CREATE:
 * -------------------------------------
 * Creates a mutex semaphore, call function in RTDS_WINDOWS.c)
 */
#define RTDS_MUTEX_SEMAPHORE_CREATE( SEMAPHORE_NAME, SEMAPHORE_NUMBER, OPTIONS ) \
    RTDS_SYSTEM_ERROR( RTDS_ERROR_CREATE_MUTEX_NOT_APPLICABLE )

/*
 * MACRO RTDS_COUNTING_SEMAPHORE_CREATE:
 * ---------------------------------------
 * Creates a counting semaphore, call function in RTDS_WINDOWS.c
 */
#define RTDS_COUNTING_SEMAPHORE_CREATE( SEMAPHORE_NAME, SEMAPHORE_NUMBER, OPTIONS, INITIAL_COUNT ) \
    RTDS_Sem_Info_Insert( SEMAPHORE_NUMBER, RTDS_SemCCreate( UITRON_SEMAPHORE_DEFAULT_MODE, INITIAL_COUNT ), RTDS_currentContext );

/*
 * MACRO RTDS_SEMAPHORE_DELETE:
 * -----------------------------
 * Delete semaphore
 */
#define RTDS_SEMAPHORE_DELETE( SEMAPHORE_ID ) \
    RTDS_SemaphoreIdDelete( SEMAPHORE_ID, RTDS_currentContext )

#define RTDS_SEMAPHORE_NAME_DELETE( SEMAPHORE_NAME, SEMAPHORE_NUMBER ) \
    RTDS_Sem_Delete( RTDS_GetSemaphoreId( SEMAPHORE_NUMBER ) );

/*
 * MACRO RTDS_SEMAPHORE_TAKE:
 * -----------------------------
 * Takes a semaphore
 */
#define RTDS_SEMAPHORE_ID_TAKE( SEMAPHORE_ID, TIME_OUT) \
    RTDS_SemaphoreIdTake( SEMAPHORE_ID, TIME_OUT, RTDS_currentContext );

#define RTDS_SEMAPHORE_NAME_TAKE( SEMAPHORE_NAME, SEMAPHORE_NUMBER, TIME_OUT ) \
    RTDS_SemaphoreIdTake( RTDS_GetSemaphoreId( SEMAPHORE_NUMBER ), TIME_OUT, RTDS_currentContext );

/*
 * MACRO RTDS_SEMAPHORE_GIVE:
 * -----------------------------
 * Gives a semaphore
 */
#define RTDS_SEMAPHORE_ID_GIVE( SEMAPHORE_ID ) \
    RTDS_SIMULATOR_TRACE(RTDS_semGive, SEMAPHORE_ID, NULL, RTDS_currentContext ); \
    RTDS_SemGive( SEMAPHORE_ID );

#define RTDS_SEMAPHORE_NAME_GIVE( SEMAPHORE_NAME, SEMAPHORE_NUMBER ) \
    RTDS_SIMULATOR_TRACE( RTDS_semGive, RTDS_GetSemaphoreId( SEMAPHORE_NUMBER ), NULL, RTDS_currentContext ); \
    RTDS_SemGive( RTDS_GetSemaphoreId( SEMAPHORE_NUMBER ) );

/* ************************************************************************************ *
 * PROCESS MACROS
 * ************************************************************************************ */

/*
 * MACRO RTDS_PROCESS_CREATE:
 * -----------------------------
 * Creates a new task representing an SDL process
 */
#define RTDS_PROCESS_CREATE( PROCESS_NAME, PROCESS_NUMBER, FUNCTION_ADDRESS, PRIORITY ) \
    RTDS_ProcessCreate( PROCESS_NUMBER, ( void * )FUNCTION_ADDRESS, PRIORITY, RTDS_HOLD, RTDS_currentContext );
/*
 * MACRO RTDS_STARTUP_PROCESS_CREATE:
 * -----------------------------
 * Used by Startup task to create a new task representing an SDL process
 */
#define RTDS_STARTUP_PROCESS_CREATE( PROCESS_NAME, PROCESS_NUMBER, FUNCTION_ADDRESS, PRIORITY ) \
    RTDS_ProcessCreate( PROCESS_NUMBER, ( void * )FUNCTION_ADDRESS, PRIORITY, RTDS_NO_HOLD, RTDS_currentContext );
/*
 * MACRO RTDS_PROCESS_KILL:
 * --------------------------
 * Kills a task representing an SDL process. An SDL task can only kill itself.
 */
#define RTDS_PROCESS_KILL \
    RTDS_ProcessKill( RTDS_currentContext );

/* ************************************************************************************ *
 * FINITE STATE MACHINE
 * ************************************************************************************ */

/*
 * MACRO RTDS_PROCEDURE_CLEAN_UP:
 * ------------------------------
 * Macro called at the end of a procedure to free last message and re-organize
 * save queue if procedure contained states.
 */
#define RTDS_PROCEDURE_CLEAN_UP \
    RTDS_TransitionCleanUp( RTDS_currentContext, 0 );

/*
 * MACRO RTDS_SDL_STATE_SET:
 * -----------------------------
 * Sets the SDL state of the current process
 */
#define RTDS_SDL_STATE_SET( NEW_SDL_STATE ) \
    RTDS_currentContext->sdlState = NEW_SDL_STATE; \
    RTDS_SIMULATOR_TRACE( RTDS_sdlStateSet, SELF, NEW_SDL_STATE, RTDS_currentContext );

/* ************************************************************************************ *
 * COVERAGE INFORMATION
 * ************************************************************************************ *
 * The coverage information is stored in an array of unsigned char. The SDL-RT symbol
 * number is used as an index; the value stored is the number of times the symbol has
 * been executed. It can not exceed 0xFF for each symbol.
 * ************************************************************************************ */
#ifdef RTDS_COVERAGE_NB_SYMBOLS
    #define RTDS_COVERAGE_DECL \
        unsigned char  RTDS_globalCoverage[ RTDS_COVERAGE_NB_SYMBOLS ];
    
    #define RTDS_COVERAGE_PROTO \
        extern RTDS_COVERAGE_DECL
    
    #define RTDS_COVERAGE_INIT \
        { \
        int RTDS_i; \
        for( RTDS_i = 0; RTDS_i < RTDS_COVERAGE_NB_SYMBOLS; RTDS_i++ ) \
            { \
            RTDS_globalCoverage[ RTDS_i ] = 0; \
            } \
        }
    
    #define RTDS_COVERAGE_LOG( SYMBOL_NUMBER ) \
        if ( RTDS_globalCoverage[ SYMBOL_NUMBER ] != 0xFF ) \
            { \
            RTDS_globalCoverage[ SYMBOL_NUMBER ]++; \
            }
#else
    #define RTDS_COVERAGE_DECL
    #define RTDS_COVERAGE_PROTO
    #define RTDS_COVERAGE_INIT
#endif

/*
 * DEFINE FOR SIMULATION:
 * ------------------------------
 * Define on how many bytes the messageUniqueId pool will be made of
 * 1 means 1 byte means 1*8 bits means 8 possible simultaneous values
 * Maximum is 8192 since the id a long type
 */
#ifdef RTDS_SIMULATOR
    #define RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE    64
#endif

#ifdef RTDS_SIMULATOR
    /* Number of levels when formatting message parameters to be printable */
    /* in the SDL-RT debugger */
    #ifndef RTDS_PARAM_CODEC_MAX_DEPTH
        #define RTDS_PARAM_CODEC_MAX_DEPTH      4
    #endif
    /* Min size of memory chunk used when formatting message parameters */
    /* to be printable in the SDL-RT debugger */
    #ifndef RTDS_PARAM_CODEC_CHUNK_SIZE
        #define RTDS_PARAM_CODEC_CHUNK_SIZE     128
    #endif
#endif

/* ************************************************************************************ *
 * CRITICAL TRACE SECTION MACROS
 * ************************************************************************************ */
#ifdef RTDS_SIMULATOR
    #define RTDS_CRITICAL_TRACE_SECTION_DECL        RTDS_SemId RTDS_globalTraceSemId;
    #define RTDS_CRITICAL_TRACE_SECTION_PROTO       extern RTDS_SemId RTDS_globalTraceSemId;
    #define RTDS_CRITICAL_TRACE_SECTION_INIT        RTDS_globalTraceSemId = RTDS_SemBCreate( RTDS_SEMAPHORE_OPTION_FIFO, 1 );
    #define RTDS_CRITICAL_TRACE_SECTION_START       RTDS_SemTake( RTDS_globalTraceSemId, RTDS_UITRON_SEM_TIMEOUT_FOREVER );
    #define RTDS_CRITICAL_TRACE_SECTION_STOP        RTDS_SemGive( RTDS_globalTraceSemId );
    #define RTDS_CRITICAL_TRACE_SECTION_POSTAMBLE   RTDS_SemDelete( RTDS_globalTraceSemId );
#else
    #define RTDS_CRITICAL_TRACE_SECTION_DECL
    #define RTDS_CRITICAL_TRACE_SECTION_PROTO
    #define RTDS_CRITICAL_TRACE_SECTION_INIT
    #define RTDS_CRITICAL_TRACE_SECTION_START
    #define RTDS_CRITICAL_TRACE_SECTION_STOP
    #define RTDS_CRITICAL_TRACE_SECTION_POSTAMBLE
#endif


/* ************************************************************************************ *
 * CRITICAL SECTION MACROS
 * ************************************************************************************ */
#define RTDS_CRITICAL_SECTION_DECL              RTDS_SemId RTDS_globalSystemSemId;
#define RTDS_CRITICAL_SECTION_PROTO             extern RTDS_SemId RTDS_globalSystemSemId;
#define RTDS_CRITICAL_SECTION_INIT              RTDS_globalSystemSemId = RTDS_SemBCreate( RTDS_SEMAPHORE_OPTION_FIFO, 1 );
#define RTDS_CRITICAL_SECTION_START             RTDS_SemTake( RTDS_globalSystemSemId, RTDS_UITRON_SEM_TIMEOUT_FOREVER );
#define RTDS_CRITICAL_SECTION_STOP              RTDS_SemGive( RTDS_globalSystemSemId );
#define RTDS_CRITICAL_SECTION_POSTAMBLE         RTDS_SemDelete( RTDS_globalSystemSemId );

/* ************************************************************************************ *
 * GLOBAL MEMORY POOL (use for dynamic memeory allocation)
 * ************************************************************************************ */
#define RTDS_GLOBAL_MEMORY_POLL_CREATION RTDS_MEMORY_POLL_CREATION

/* ************************************************************************************ *
 * STARTUP SYNCHRONISATION
 * ************************************************************************************ */
#define RTDS_HOLD                           1
#define RTDS_NO_HOLD                        0
#define RTDS_START_SYNCHRO_DECL             RTDS_SemId RTDS_globalStartSynchro;
#define RTDS_START_SYNCHRO_PROTO            extern RTDS_SemId RTDS_globalStartSynchro;
#define RTDS_START_SYNCHRO_INIT             RTDS_globalStartSynchro = RTDS_SemCCreate( UITRON_SEMAPHORE_DEFAULT_MODE, 0 );
#define RTDS_START_SYNCHRO_WAIT \
                                            RTDS_SemTake( RTDS_globalStartSynchro, RTDS_UITRON_SEM_TIMEOUT_FOREVER ); \
                                            RTDS_SemGive( RTDS_globalStartSynchro );

#define RTDS_START_SYNCHRO_GO \
                                            RTDS_SemFlush( RTDS_globalStartSynchro ); \
                                            RTDS_SemGive( RTDS_globalStartSynchro );

#define RTDS_START_SYNCHRO_POSTAMBLE        RTDS_SemDelete( RTDS_globalStartSynchro );

#ifdef RTDS_SIMULATOR
    #define RTDS_START_SYNCHRO_HOLD         RTDS_SemTake( RTDS_globalStartSynchro, RTDS_UITRON_SEM_TIMEOUT_FOREVER );
    #define RTDS_START_SYNCHRO_UNHOLD       RTDS_SemGive( RTDS_globalStartSynchro );
#else
    #define RTDS_START_SYNCHRO_HOLD
    #define RTDS_START_SYNCHRO_UNHOLD
#endif /* RTDS_SIMULATOR */

#if defined( RTDS_SIMULATOR )
    #define RTDS_SIMULATOR_TRACE( EVENT, PARAM1, PARAM2, RTDS_CURRENT_CONTEXT ) \
        RTDS_CRITICAL_TRACE_SECTION_START; \
        if( ( EVENT == RTDS_messageSent ) || ( EVENT == RTDS_messageReceived ) ) \
            { \
            RTDS_messageDataToString( &RTDS_globalPrintableParameters, ( ( RTDS_MessageHeader * )PARAM1 )->messageNumber, ( ( RTDS_MessageHeader * )PARAM1 )->dataLength, ( void * )( ( ( RTDS_MessageHeader * )PARAM1 )->pData ), RTDS_PARAM_CODEC_MAX_DEPTH ); \
            } \
        RTDS_globalTraceEntry.event = EVENT; \
        RTDS_globalTraceEntry.eventParameter1 = ( void * )PARAM1; \
        RTDS_globalTraceEntry.eventParameter2 = ( long )PARAM2; \
        RTDS_globalTraceEntry.currentContext = ( RTDS_GlobalProcessInfo * )RTDS_CURRENT_CONTEXT; \
        RTDS_SYS_TIME_REFRESH; \
        RTDS_DummyTraceFunction(); \
        if( ( EVENT == RTDS_messageSent ) || ( EVENT == RTDS_messageReceived ) ) \
            { \
            RTDS_FREE( RTDS_globalPrintableParameters ); \
            RTDS_globalPrintableParameters = NULL; \
            } \
        RTDS_CRITICAL_TRACE_SECTION_STOP; 
#else
    #define RTDS_SIMULATOR_TRACE( EVENT, PARAM1, PARAM2, RTDS_CURRENT_CONTEXT )
#endif

/*
 * MACRO RTDS_SYSTEM_ERROR:
 * ------------------------
 * Macro called when a system error is detected in RTDS 'kernel'
 * Needs to be defined by user.
 *
 * For printing in the std output include "stdio.h" and remove this comment
 * printf("RTDS error 0x%x\nCheck RTDS_Error.h file for explanations.\n",errorNumber); \
 */
#ifndef RTDS_SYSTEM_ERROR
    #define RTDS_SYSTEM_ERROR( errorNumber ) \
        { \
        RTDS_SIMULATOR_TRACE( RTDS_systemError, errorNumber, RTDS_SYS_NONE, NULL ); \
        };
#endif

/*
 * DEFINE FOR QUEUE MANIPULATION:
 * ------------------------------
 */
#define RTDS_QUEUE_MAX_MSG                  20                              /* Maximum number of messages in a queue */
#define RTDS_QUEUE_MAX_MSG_LENGTH           sizeof( RTDS_MessageHeader )    /* Maximum length of messages in a queue */

/*
 * DEFINE FOR TIMER MANIPULATION:
 * ------------------------------
 */
#define RTDS_TIMER_OK               1
#define RTDS_TIMER_NA               2
#define RTDS_TIMER_CANCELLED        0

#define RTDS_SYS_TIME_DECL \
    long RTDS_globalSystemTime = 0; \
    long RTDS_globalStartTime = 0;

#define RTDS_SYS_TIME_PROTO \
    extern long RTDS_globalSystemTime; \
    extern long RTDS_globalStartTime;

#define RTDS_SYS_TIME_INIT \
    RTDS_globalStartTime = RTDS_TickGet();

#define RTDS_SYS_TIME_REFRESH \
    RTDS_globalSystemTime = RTDS_TickGet() - RTDS_globalStartTime;

#define RTDS_SYS_CURRENT_TASKID_REFRESH \
    RTDS_GetCurrentThread();

/*
 * GLOBAL VARIABLE
 * ----------------
 *
 */
RTDS_COVERAGE_PROTO
RTDS_SYS_TIME_PROTO
RTDS_CRITICAL_SECTION_PROTO
RTDS_START_SYNCHRO_PROTO
RTDS_CRITICAL_TRACE_SECTION_PROTO

#ifdef __cplusplus
}
#endif

#endif /* End of _RTDS_MACRO_H_ */
