/* PragmaDev RTDS FreeRTOS integration */
/* Integration with FreeRTOS RTOS      */

#include "RTDS_OS.h"
#include "RTDS_MACRO.h"


/* **************************************************************** *
 * List of functions defined in this file
 * **************************************************************** *
 *	RTDS_GetTimerUniqueId
 *  RTDS_WatchDogFunction
 *	RTDS_StartTimer
 *	RTDS_StopTimer
 *	RTDS_GetProcessQueueId
 *	RTDS_MsgQueueSend
 *	RTDS_SimulatorMsgQueueSend
 *	RTDS_MsgQueueReceive
 *	RTDS_ProcessCreate
 *	RTDS_ProcessKill
 *	RTDS_Sem_Info_Insert
 *	RTDS_Sem_Delete
 *	RTDS_BinarySemaphoreCreate
 *	RTDS_MutexSemaphoreCreate
 *	RTDS_CountingSemaphoreCreate
 *	RTDS_SemaphoreIdTake
 *	RTDS_GetSemaphoreId
 *	RTDS_GetSystemTime
 *	RTDS_GetMessageUniqueId
 *	RTDS_ReleaseMessageUniqueId
 *	RTDS_MemAlloc
 *	RTDS_DummyTraceFunction
 * **************************************************************** */

 

/* **************************************************************** *
 *	RTDS_GetTimerUniqueId
 * **************************************************************** *
 * Get an available timer unique id.
 * Is used to set timers
 * **************************************************************** *
 * Parameters:
 *	 timerList: points to the first element of
 * the chained list of RTDS_TimerState
 * Returns:
 *	 The unique timer Id
 * **************************************************************** *
 * Context:
 *	 The chained list of RTDS_TimerState is sorted by increasing
 *	 unique timer id
 *	 timerUniqueId = NULL means the message is not a timer; so it
 *	 is avoided.
 * **************************************************************** */

long RTDS_GetTimerUniqueId(RTDS_TimerState *timerList)
	{
	RTDS_TimerState *RTDS_prevTimer,*RTDS_timer;
	long newTimerId;

	RTDS_CRITICAL_SECTION_START;
	/* If list is empty take 1 */
	if ( timerList == NULL )
		newTimerId = 1;
	/* If 1 is available, take it */
	else if ( timerList->timerUniqueId != 1 )
		newTimerId = 1;
	else
		{
		/* If there is a gap in the timerId chained list, take an Id in the gap */
		RTDS_prevTimer = timerList;
		newTimerId = 0;
		for ( RTDS_timer = timerList->next ; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
			{
			if ( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 ) newTimerId = RTDS_prevTimer->timerUniqueId + 1;
			RTDS_prevTimer = RTDS_timer;
			}
		/* No gap, let's take the next value */
		if ( newTimerId == 0 ) newTimerId = RTDS_prevTimer->timerUniqueId + 1;
		}

	RTDS_CRITICAL_SECTION_STOP;
	/* Check the counter did not go back to 0 */
	if ( newTimerId == 0 )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID);
	return newTimerId;
	}

    
    
/* **************************************************************** *
 * RTDS_WatchDogFunction
 * **************************************************************** *
 * Send a timer message using the parameters given to the watchdog
 * **************************************************************** *
 * Parameters:
 *      xTimerHandle: Needed parameters to create and send the timer:
 *          WatchDogId
 * Returns:
 *      Nothing
 * **************************************************************** *
 * Context:
 *     Each time a timer is set, a watchdog is created; that's why it
 *     has to destroy itself.
 * **************************************************************** */
void RTDS_WatchDogFunction( xTimerHandle pxTimer )
    {
    RTDS_MessageHeader      *newMessage;
    RTDS_TimerState         *timerState, *parameter;
    RTDS_GlobalProcessInfo	 *processInfo;
          
    /* Suspend tasks otherwise multiple calls to watchdog will raise a sigtrap */
    vTaskSuspendAll();
    
    parameter = NULL;
    
    /* Let's find the timer information for this watchdog */
    for (processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next)
        for (timerState = processInfo->timerList ; timerState != NULL ; timerState = timerState->next)
            {
            if ( timerState->watchDogId == pxTimer)
                {
                parameter = timerState;
                break;
                }
            }
        
    
    if ( parameter == NULL)
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_NOT_FOUND );
        }
        
    /* Fill in the message parameters */
    newMessage = (RTDS_MessageHeader *)RTDS_MALLOC(sizeof(RTDS_MessageHeader));
    #ifdef RTDS_SIMULATOR
        newMessage->messageUniqueId = 0;
    #endif
    newMessage->messageNumber = parameter->timerNumber;
    newMessage->timerUniqueId = parameter->timerUniqueId;
    newMessage->receiver = parameter->receiverSdlInstanceId;
    newMessage->sender = parameter->receiverSdlInstanceId;
    newMessage->dataLength = 0;
    newMessage->pData = NULL;
    newMessage->next = NULL;
    
    /* Send the message */
    if( xQueueSend( parameter->receiverSdlInstanceId->queueId, ( char * )&newMessage, portMAX_DELAY ) == ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_Q_SEND_IN_WATCHDOG );
        }
        
    /* We have to release all memory hold by the watchdog */
    /* So we set its state to finished */
    parameter->state = RTDS_TIMER_FINISHED;
    
    xTaskResumeAll();
    
    }

    

/* **************************************************************** *
 *	RTDS_StartTimer
 * **************************************************************** *
 * Starts a cyclic timer with the necessary parameters to create
 * the timer when it goes off.
 * **************************************************************** *
 * Parameters:
 *	- InstanceId of the receiver
 *	- TimerNumber
 *	- TimerUniqueId
 *	- delay of the timer
 *	- timerStateList: Address of the list of timer
 *	- RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_StartTimer(
	RTDS_SdlInstanceId     * instanceId,
	long                     timerNumber,
	long                     timerUniqueId,
	int                      delay,
	RTDS_TimerState       ** pTimerStateList,
	RTDS_GlobalProcessInfo * RTDS_currentContext)
	{
	xTimerHandle            watchDogId;
	RTDS_TimerState         *timerState,*previousTimerState,*newTimerState;

  /* Create the timer. */
  watchDogId = xTimerCreate( 
    "RTDS Timer Name",            /* Timer name */
    delay,                        /* Delay in ticks */
    pdFALSE,                      /* One shot timer */
    (void *)timerUniqueId,        /* Timer unique identifier */
    RTDS_WatchDogFunction);       /* Callback function */
	if ( watchDogId == NULL )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_TIMER_CREATE);

    
    
	/* Update the list of timers */
	/* Important note: it is a sorted list based on the timerUniqueId field */
	newTimerState = (RTDS_TimerState *)RTDS_MALLOC(sizeof(RTDS_TimerState));
	if (newTimerState == NULL)
		RTDS_SYSTEM_ERROR(RTDS_ERROR_MALLOC_TIMER_STATE_IN_START_TIMER);

	/* Initialize the new element */
	newTimerState->state                    = RTDS_TIMER_OK;
	newTimerState->timerNumber              = timerNumber;
	newTimerState->timeoutValue             = delay + xTaskGetTickCount(); /* Only way found to follow up the timer */
	newTimerState->timerUniqueId            = timerUniqueId;
	newTimerState->next                     = NULL;
	newTimerState->watchDogId               = watchDogId;
	newTimerState->receiverSdlInstanceId    = instanceId;

	/* Insert the new element */
	if (*pTimerStateList == NULL) /* The list is empty */
		*pTimerStateList = newTimerState;
	else	/* The list is not empty */
		{
		previousTimerState = NULL;
		for (timerState = *pTimerStateList ; timerState != NULL ; timerState = timerState->next)
			{
			if (timerState->timerUniqueId > timerUniqueId)
				{
				if (previousTimerState == NULL)
					{
					*pTimerStateList = newTimerState;
					newTimerState->next = timerState;
					}
				else
					{
					previousTimerState->next = newTimerState;
					newTimerState->next = timerState;
					}
				break;
				}
			previousTimerState = timerState;
			}
		if (timerState == NULL) /* Inserted at the end of the list */
			previousTimerState->next = newTimerState;
		}

	RTDS_SIMULATOR_TRACE(RTDS_timerStarted, newTimerState, delay, RTDS_currentContext);

	/* Start the timer */
	if ( xTimerStart(watchDogId, 0) != pdPASS  )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_TIMER_START);
}



/* **************************************************************** *
 *	RTDS_StopTimer
 * **************************************************************** *
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 * **************************************************************** *
 * Parameters:
 *	- TimerNumber
 *	- pTimerStateList pointing to the timer chained list
 *	- RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_StopTimer(
	long										timerNumber,
	RTDS_TimerState 				**pTimerStateList,
	RTDS_GlobalProcessInfo	*RTDS_currentContext)
	{
	RTDS_TimerState 	 *RTDS_timer, *RTDS_prevTimer;
	RTDS_MessageHeader *RTDS_message, *RTDS_prevMessage;


	RTDS_prevTimer = NULL;
	for ( RTDS_timer = *pTimerStateList ; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
		{
		if (( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ))
			{
			/* Let's try to cancel the timer */
			/* Watch out some RTOS might return ok even if the timer went off... */
			if ( xTimerStop(RTDS_timer->watchDogId, 0) == pdPASS  )
				{
				if (RTDS_prevTimer == NULL)
					{
					*pTimerStateList = RTDS_timer->next;
					}
				/* Not the first of the list */
				else
					{
					RTDS_prevTimer->next = RTDS_timer->next;
					}

				/* Free the timerEvent array */
				RTDS_FREE(RTDS_timer);
				}
			/* Could not cancel the timer. Probably went off allready */
			/* Set it cancelled in the list */
			else
				{
				RTDS_timer->state = RTDS_TIMER_CANCELLED;
				}

			RTDS_SIMULATOR_TRACE(RTDS_timerCancelled, RTDS_timer, NULL, RTDS_currentContext);
			return;
			}
		RTDS_prevTimer = RTDS_timer;
		}

	/* If execution gets here: the timer might be in the save queue */
	RTDS_prevMessage = NULL;
	for ( RTDS_message=RTDS_currentContext->readSaveQueue ; RTDS_message!=NULL ; RTDS_message=RTDS_message->next )
		{
		if ( RTDS_message->messageNumber == timerNumber )
			{
			/* Remove it from the list */
			/* Is the first of the list */
			if (RTDS_prevMessage == NULL)
				{
				RTDS_currentContext->readSaveQueue = RTDS_message->next;
				}
			/* Not the first of the list */
			else
				{
				RTDS_prevMessage->next = RTDS_message->next;
				}
			RTDS_FREE(RTDS_message);
			return;
			}
		RTDS_prevMessage = RTDS_message;
		}

	/* If execution gets here: the timer might be in the save queue */
	RTDS_prevMessage = NULL;
	for ( RTDS_message=RTDS_currentContext->writeSaveQueue ; RTDS_message!=NULL ; RTDS_message=RTDS_message->next )
		{
		if ( RTDS_message->messageNumber == timerNumber )
			{
			/* Remove it from the list */
			/* Is the first of the list */
			if (RTDS_prevMessage == NULL)
				{
				RTDS_currentContext->writeSaveQueue = RTDS_message->next;
				}
			/* Not the first of the list */
			else
				{
				RTDS_prevMessage->next = RTDS_message->next;
				}
			RTDS_FREE(RTDS_message);
			return;
			}
		RTDS_prevMessage = RTDS_message;
		}

	}



/* **************************************************************** *
 *	RTDS_GetProcessQueueId
 * **************************************************************** *
 * Returns the queueId of process based on its name
 * **************************************************************** *
 * Parameters:
 *		 - process name as a number defined in RTDS_gen.h
 * Returns:
 *	 the process queue id if found
 * Error:
 *	 System error call if not found and NULL returned
 * **************************************************************** */

RTDS_SdlInstanceId * RTDS_GetProcessQueueId(int processNumber)
	{
	RTDS_GlobalProcessInfo	*processInfo;
	RTDS_QueueId			foundQueue = NULL;

	RTDS_CRITICAL_SECTION_START;
	for (processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next)
		{
		if ( processInfo->sdlProcessNumber == processNumber )
			{
			foundQueue = processInfo->mySdlInstanceId;
			break;
			}
		}
	RTDS_CRITICAL_SECTION_STOP;

	if (foundQueue == NULL)
		RTDS_SYSTEM_ERROR(RTDS_ERROR_GET_PROCESS_QUEUE_ID);

	return foundQueue;
	}



/* **************************************************************** *
 *	RTDS_MsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue
 * **************************************************************** *
 * Parameters:
 *	- messageNumber representing a message name
 *	- dataLength length of data pointed by pData
 *	- pData  pointer on data sent with the message
 *	- receiver message receiver queue address
 *	- sender	 message sender queue address
 *	- RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_MsgQueueSend(
		long                    messageNumber,
		long                    dataLength,
		unsigned char           *pData,
		RTDS_SdlInstanceId      *receiver,
		RTDS_SdlInstanceId      *sender,
		RTDS_GlobalProcessInfo	*RTDS_currentContext )
	{
    RTDS_MessageHeader *RTDS_messageToSend;
    
    RTDS_messageToSend = (RTDS_MessageHeader *)RTDS_MALLOC(sizeof(RTDS_MessageHeader));
    RTDS_messageToSend->messageNumber = messageNumber;
    RTDS_messageToSend->timerUniqueId = 0;
    RTDS_messageToSend->sender = sender;
    RTDS_messageToSend->receiver = receiver;
    RTDS_messageToSend->dataLength = dataLength;
    RTDS_messageToSend->pData = pData;
    RTDS_messageToSend->next = NULL;
    
    #ifdef RTDS_SIMULATOR
        RTDS_CRITICAL_TRACE_SECTION_START
        RTDS_messageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
        RTDS_messageDataToString( &RTDS_globalPrintableParameters, messageNumber, dataLength, ( void * )pData, RTDS_PARAM_CODEC_MAX_DEPTH );
        RTDS_SIMULATOR_TRACE( RTDS_messageSent, RTDS_messageToSend, receiver, RTDS_currentContext );
        RTDS_FREE( RTDS_globalPrintableParameters );
        RTDS_globalPrintableParameters = NULL;
        RTDS_CRITICAL_TRACE_SECTION_STOP
    #endif

	if ( xQueueSend(receiver->queueId,(void *)(&RTDS_messageToSend), portMAX_DELAY) != pdPASS  )
		RTDS_SYSTEM_ERROR(RTDS_ERROR_MSG_QUEUE_SEND);
	}


/* **************************************************************** *
 * RTDS_MsgQueueReceive
 * **************************************************************** *
 * Receives a message from a process's queue
 * **************************************************************** *
 * Parameters:
 *          queue id to read the messsage from
 *          pointer on the message to receive
 *          size of message
 *          mode used to read (blocking or non blocking)
 * Returns:
 *          RTDS_OK if everything goes well, RTDS_ERROR otherwise
  * **************************************************************** */
int RTDS_MsgQueueReceive( RTDS_SdlInstanceId * sdlInstanceId, unsigned char * message, int mode )
    {
    if( xQueueReceive( sdlInstanceId->queueId, (void *)message, portMAX_DELAY ) != pdTRUE  )
        {
        return RTDS_ERROR;
        }
    return RTDS_OK;
    }




/* **************************************************************** *
 *	RTDS_ProcessCreate
 * **************************************************************** *
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 * **************************************************************** *
 * Parameters:
 *		 - name of the process as a number
 *		 - address of the process function
 *		 - priority fo the process
 *		 - the address of the address of the msg queue to be created
 *		 - the address of the msg queue of the parent process
 *		 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing but updates the OFFSPRING of the caller
 * **************************************************************** */

void RTDS_ProcessCreate(
  char                    *processName,
	int                     processNumber,
	void                    (*functionAddress)(RTDS_GlobalProcessInfo*),
	int                     priority,
	RTDS_SdlInstanceId      ** pOffspringSdlInstanceId,
	RTDS_SdlInstanceId      * selfQueueId,
	short                   synchronization,
	RTDS_GlobalProcessInfo  * RTDS_currentContext )
	{
	RTDS_GlobalProcessInfo  *processInfo, *newProcessInfo;

	/* Allocate and fill in a new processInfo structure */
	newProcessInfo = (RTDS_GlobalProcessInfo *)RTDS_MALLOC(sizeof(RTDS_GlobalProcessInfo));
	if ( newProcessInfo == NULL )
			RTDS_SYSTEM_ERROR(RTDS_ERROR_PROCESS_CREATE_MALLOC);

	/* Sets the process name as a number */
	newProcessInfo->sdlProcessNumber        = processNumber;
	/* SDL initial state */
	newProcessInfo->sdlState                = 0;

	/* Last structure of the chained list */
	newProcessInfo->next                    = NULL;
	newProcessInfo->parentSdlInstanceId     = selfQueueId;
	newProcessInfo->offspringSdlInstanceId  = NULL;
	newProcessInfo->currentMessage          = NULL;
	newProcessInfo->timerList               = NULL;
#ifdef RTDS_SIMULATOR
	newProcessInfo->priority                = priority;
#endif
    *pOffspringSdlInstanceId = ( RTDS_SdlInstanceId * )RTDS_MALLOC( sizeof( RTDS_SdlInstanceId ) );
    /* New task's queue creation */
    ( *pOffspringSdlInstanceId )->queueId = xQueueCreate( (unsigned portBASE_TYPE)RTDS_QUEUE_MAX_MSG,(unsigned portBASE_TYPE)(RTDS_QUEUE_MAX_MSG_LENGTH));
    if ( ( *pOffspringSdlInstanceId )->queueId == 0 )
        {
        RTDS_SYSTEM_ERROR(RTDS_ERROR_MSG_QUEUE_CREATE);
        }
    ( *pOffspringSdlInstanceId )->instanceNumber = 0;
    newProcessInfo->mySdlInstanceId = *pOffspringSdlInstanceId;
    *pOffspringSdlInstanceId = newProcessInfo->mySdlInstanceId;


	/* Take the synchronization semaphore so that the created task does not run straight away */
	if (synchronization == RTDS_HOLD )
		RTDS_START_SYNCHRO_HOLD;
	/* Task creation */
	if ( xTaskCreate( 
        (pdTASK_CODE) functionAddress, 
        processName, 
        (unsigned portSHORT) RTDS_TASK_STACK_SIZE, 
        (void *) newProcessInfo, 
        (unsigned portBASE_TYPE) priority, 
        (xTaskHandle *)&(newProcessInfo->myRtosTaskId) 
        ) != pdPASS  )
            {
            RTDS_SYSTEM_ERROR(RTDS_ERROR_PROCESS_CREATE_TASK_SPAWN);
            }
                          
	/* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
	RTDS_CRITICAL_SECTION_START;
	if ( RTDS_globalProcessInfo == NULL )
		RTDS_globalProcessInfo = newProcessInfo;
	else
		{
		/* Let's get to the end of the list */
		for (processInfo = RTDS_globalProcessInfo ; processInfo->next != NULL ; processInfo = processInfo->next) ;
		processInfo->next = newProcessInfo;
		}

	RTDS_CRITICAL_SECTION_STOP;

	RTDS_SIMULATOR_TRACE(RTDS_processCreated, newProcessInfo, NULL, RTDS_currentContext);

	/* The newly created task can now run: RTDS_globalProcessInfo and trace are up to date. */
	if (synchronization == RTDS_HOLD )
		RTDS_START_SYNCHRO_UNHOLD;
	}




/* **************************************************************** *
 *	RTDS_ProcessKill
 * **************************************************************** *
 * Kills an SDL process and delete its queue and process info block
 * **************************************************************** *
 * Parameters:
 *	- RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_ProcessKill( RTDS_GlobalProcessInfo  *RTDS_currentContext )
	{
	RTDS_GlobalProcessInfo	  *processInfo, *previousProcessInfo;
	RTDS_TimerState 				     *RTDS_timer;
	RTDS_MessageHeader       *message, *tmpMessage;
	xTaskHandle              pidToKill=NULL;

	/* Let's free the current message if any */
	if (RTDS_currentContext->currentMessage != NULL )
		{
    /* Free the message parameter structure if any */
    if (RTDS_currentContext->currentMessage->pData != NULL){
      RTDS_FREE(RTDS_currentContext->currentMessage->pData);
      }

		#ifdef RTDS_SIMULATOR
			/* Release the message unique id back to the pool */
			RTDS_ReleaseMessageUniqueId(RTDS_currentContext->currentMessage->messageUniqueId);
		#endif

		/* Free memory */
		RTDS_FREE(RTDS_currentContext->currentMessage);
		}

	/* Let's clean the timer chained list to free memory and stop watchdogs */
	for ( RTDS_timer = RTDS_currentContext->timerList ; RTDS_timer != NULL ; RTDS_timer = RTDS_currentContext->timerList)
		{
		xTimerStop(RTDS_timer->watchDogId, 0);
		RTDS_currentContext->timerList = RTDS_timer->next;
		RTDS_FREE(RTDS_timer);
		}

	/* Clean the save queue: free messages and message unique ids in the read and write save queues */
	for (message=RTDS_currentContext->readSaveQueue;message!=NULL;)
	{
	#ifdef RTDS_SIMULATOR
		RTDS_ReleaseMessageUniqueId(message->messageUniqueId);
	#endif
	tmpMessage = message;
	message = message->next;
	RTDS_FREE(tmpMessage);
	}

	for (message=RTDS_currentContext->writeSaveQueue;message!=NULL;)
	{
	#ifdef RTDS_SIMULATOR
		RTDS_ReleaseMessageUniqueId(message->messageUniqueId);
	#endif
	tmpMessage = message;
	message = message->next;
	RTDS_FREE(tmpMessage);
	}

	previousProcessInfo = NULL;
	RTDS_CRITICAL_SECTION_START;
	for (processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next)
		{
		/* The queue id is used to find the process id */
		if ( processInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )  /* Found ! */
			{
			/* Trace is done now otherwise the Simulator can not find out which task dies */
			RTDS_SIMULATOR_TRACE(RTDS_processDied, processInfo, NULL, RTDS_currentContext);

			/* Update the process information chained list */
			if ( previousProcessInfo == NULL )	/* First one in the list */
				RTDS_globalProcessInfo = processInfo->next;
			else
				previousProcessInfo->next = processInfo->next;

			pidToKill = processInfo->myRtosTaskId;
			/* Delete the message queue and free the process info block */
			RTDS_FREE(processInfo);
			break;
			}
		previousProcessInfo = processInfo;
		}

	RTDS_CRITICAL_SECTION_STOP;

	/* Delete the task */
	if ( pidToKill != NULL)
		{
		vTaskDelete(pidToKill);
		}
	else
		RTDS_SYSTEM_ERROR(RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND);

	}






/* **************************************************************** *
 *	RTDS_Sem_Info_Insert
 * **************************************************************** *
 * Inserts a semaphoreInfo struct in the semaphore info chained list
 * **************************************************************** *
 * Parameters:
 *	- semaphore number representing its name
 *	- id of the semaphore
 *	- RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_Sem_Info_Insert(
		int                     semaphoreNumber,
		RTDS_SemaphoreId        semaphoreId,
		RTDS_GlobalProcessInfo	*RTDS_currentContext)
	{
	RTDS_GlobalSemaphoreInfo	*semInfo, *newSemInfo;
  long semIsAvailable = -1;

	/* Allocate and fill in a new semInfo structure */
	newSemInfo = (RTDS_GlobalSemaphoreInfo *)RTDS_MALLOC(sizeof(RTDS_GlobalSemaphoreInfo));
	if ( newSemInfo == NULL )
			RTDS_SYSTEM_ERROR(RTDS_ERROR_SEM_INFO_INSERT_MALLOC);

	newSemInfo->semaphoreNumber = semaphoreNumber;
	newSemInfo->next            = NULL;

	/* Semaphore creation */
	newSemInfo->semaphoreId = semaphoreId;
	if ( newSemInfo->semaphoreId == NULL )
			RTDS_SYSTEM_ERROR(RTDS_ERROR_SEM_INFO_INSERT);

	/* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
	RTDS_CRITICAL_SECTION_START;
	if ( RTDS_globalSemaphoreInfo == NULL )
		RTDS_globalSemaphoreInfo = newSemInfo;
	else
		{
		/* Let's get to the end of the list */
		for (semInfo = RTDS_globalSemaphoreInfo ; semInfo->next != NULL ; semInfo = semInfo->next) ;
		semInfo->next = newSemInfo;
		}
	RTDS_CRITICAL_SECTION_STOP;

  #if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
      semIsAvailable=RTDS_IsSemaphoreAvailable(semaphoreId);
  #endif

	RTDS_SIMULATOR_TRACE(RTDS_semaphoreCreated, semaphoreId, semIsAvailable, RTDS_currentContext);
	}





/* **************************************************************** *
 *	RTDS_Sem_Delete
 * **************************************************************** *
 * Deletes a semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 * **************************************************************** *
 * Parameters:
 *		 - the address of the semaphore to find the semaphore info
 *			 block
 *		 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_Sem_Delete( RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo	*RTDS_currentContext )
	{
	RTDS_GlobalSemaphoreInfo *semInfo, *semInfoPrev;

	RTDS_SIMULATOR_TRACE(RTDS_semaphoreDeleted, semaphoreId, 0, RTDS_currentContext);

	/* Remove the semaphore information from the chained list */
	semInfoPrev = NULL;
	RTDS_CRITICAL_SECTION_START;
	for (semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next)
		{
		if ( semInfo->semaphoreId == semaphoreId) /* semaphore found */
			{
			if (semInfoPrev == NULL)
				RTDS_globalSemaphoreInfo = semInfo->next;
			else
				semInfoPrev->next = semInfo->next;

			/* 1 option value will always flush the semaphore. Owned or not */
			xSemaphoreGive(semInfo->semaphoreId);

			RTDS_FREE(semInfo);
			break;
			}
		semInfoPrev = semInfo;
		}
	RTDS_CRITICAL_SECTION_STOP;

	}



/* **************************************************************** *
 *	RTDS_BinarySemaphoreCreate
 * **************************************************************** *
 * Creates a mutex semaphore
 * **************************************************************** *
 * Parameters:
 *	 - semaphore number representing its name
 *	 - semaphore initial state
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 - semaphoreId
 * **************************************************************** */

RTDS_SemaphoreId	RTDS_BinarySemaphoreCreate(
		int 										semaphoreNumber,
		unsigned char 					initialState,
		RTDS_GlobalProcessInfo	*RTDS_currentContext)
	{
	RTDS_SemaphoreId	RTDS_semaphoreId;

  vSemaphoreCreateBinary(RTDS_semaphoreId);

  /* Default state is available. */
  if (initialState == 0)
      xSemaphoreTake(RTDS_semaphoreId, 0);
      
	if (RTDS_semaphoreId != NULL)
		RTDS_Sem_Info_Insert( semaphoreNumber, RTDS_semaphoreId, RTDS_currentContext );
	else
		{
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEMAPHORE_CREATE);
		}

	return RTDS_semaphoreId;
	}



/* **************************************************************** *
 *	RTDS_MutexSemaphoreCreate
 * **************************************************************** *
 * Creates a mutex semaphore
 * **************************************************************** *
 * Parameters:
 *	 - semaphore number representing its name
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 - semaphoreId
 * **************************************************************** */

RTDS_SemaphoreId RTDS_MutexSemaphoreCreate(int semaphoreNumber, RTDS_GlobalProcessInfo	*RTDS_currentContext)
	{
	RTDS_SemaphoreId	RTDS_semaphoreId;

	RTDS_semaphoreId = xSemaphoreCreateMutex();
	if ( RTDS_semaphoreId != NULL)
		RTDS_Sem_Info_Insert( semaphoreNumber, RTDS_semaphoreId, RTDS_currentContext );
	else
		{
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEMAPHORE_CREATE);
		}

	return RTDS_semaphoreId;
	}



/* **************************************************************** *
 *	RTDS_CountingSemaphoreCreate
 * **************************************************************** *
 * Creates a counting semaphore
 * **************************************************************** *
 * Parameters:
 *	 - semaphore number representing its name
 *	 - initial semaphore count value
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 - semaphoreId
 * **************************************************************** */

RTDS_SemaphoreId RTDS_CountingSemaphoreCreate(
		int 										semaphoreNumber,
		unsigned char 					initialCount,
		RTDS_GlobalProcessInfo	*RTDS_currentContext)
  {
	RTDS_SemaphoreId	RTDS_semaphoreId;

  RTDS_semaphoreId = xSemaphoreCreateCounting( (unsigned portBASE_TYPE) RTDS_BINARY_MAX_COUNT, (unsigned portBASE_TYPE) initialCount );
	if (RTDS_semaphoreId!=NULL)
		RTDS_Sem_Info_Insert( semaphoreNumber, RTDS_semaphoreId, RTDS_currentContext );
	else
		{
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEMAPHORE_CREATE);
		}

	return RTDS_semaphoreId;
	}



/* **************************************************************** *
 *	RTDS_SemaphoreIdTake
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *	 - the id of the semaphore
 *	 - timeout value
 *	 - RTDS_currentContext currentContext of the task
 * Returns:
 *	 - status
 * **************************************************************** */

RTDS_SemaphoreStatus RTDS_SemaphoreIdTake(
	RTDS_SemaphoreId					semaphoreId,
	RTDS_SemaphoreTimeout 		TIME_OUT,
	RTDS_GlobalProcessInfo		*RTDS_currentContext )
		{
		RTDS_SemaphoreStatus		tmpStatus;
    long semIsAvailable = -1;

		RTDS_SIMULATOR_TRACE(RTDS_semTakeAttempt, semaphoreId, TIME_OUT, RTDS_currentContext);

		if (TIME_OUT == RTDS_SEMAPHORE_TIME_OUT_NO_WAIT)
			tmpStatus = xSemaphoreTake(semaphoreId, 0);
		else if (TIME_OUT == RTDS_SEMAPHORE_TIME_OUT_FOREVER)
			tmpStatus = xSemaphoreTake( semaphoreId, portMAX_DELAY);
		else
			tmpStatus = xSemaphoreTake( semaphoreId, TIME_OUT);

		if ( tmpStatus == pdTRUE  )
				{
				tmpStatus = RTDS_OK;
        #if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
            semIsAvailable=RTDS_IsSemaphoreAvailable(semaphoreId);
        #endif
  				RTDS_SIMULATOR_TRACE(RTDS_semTakeSucceded, semaphoreId, semIsAvailable, RTDS_currentContext);
				}
		else if ( tmpStatus == pdFALSE )
				{
				tmpStatus = RTDS_ERROR;
				RTDS_SIMULATOR_TRACE(RTDS_semTakeTimedOut, semaphoreId, NULL, RTDS_currentContext);
				}
		else
				{
				RTDS_SYSTEM_ERROR(RTDS_ERROR_SEMAPHORE_ID_TAKE);
				}

		return tmpStatus;
		}



/* **************************************************************** *
 *	RTDS_GetSemaphoreId
 * **************************************************************** *
 * Gets the id of a semaphore from its number (name)
 * **************************************************************** *
 * Parameters:
 *	 - semaphore number representing its name
 * Returns:
 *	 - the id of the semaphore
 * **************************************************************** */

RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
	{
	RTDS_GlobalSemaphoreInfo	*semInfo;
	RTDS_SemaphoreId			foundSemaphoreId;

	foundSemaphoreId = NULL;
	RTDS_CRITICAL_SECTION_START;
	for (semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next)
		{
		if ( semInfo->semaphoreNumber == semaphoreNumber ) /* semaphore found */
			{
			foundSemaphoreId = semInfo->semaphoreId;
			break;
			}
		}
	RTDS_CRITICAL_SECTION_STOP;

	if (foundSemaphoreId==NULL)
		RTDS_SYSTEM_ERROR(RTDS_ERROR_GET_SEMAPHORE_ID);

	return foundSemaphoreId;	/* Might cause an RTOS exception if NULL*/
	}



/* **************************************************************** *
 *	RTDS_TransitionCleanUp
 * **************************************************************** *
 * Called at the end of transitions:
 * - frees message buffer if valid
 * - re-organize save queue if state has changed
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 Nothing
 * **************************************************************** */

void RTDS_TransitionCleanUp(RTDS_GlobalProcessInfo *RTDS_currentContext, int RTDS_sdlStatePrev)
	{
	/* Free message buffer if valid */
	if (RTDS_currentContext->currentMessage != NULL )
		{
    /* Free the message parameter structure if any */
    if (RTDS_currentContext->currentMessage->pData != NULL){
      RTDS_FREE(RTDS_currentContext->currentMessage->pData);
      }

		#ifdef RTDS_SIMULATOR
			/* Release the message unique id back to the pool */
			RTDS_ReleaseMessageUniqueId(RTDS_currentContext->currentMessage->messageUniqueId);
		#endif

		/* Free memory */
		RTDS_FREE(RTDS_currentContext->currentMessage);
		RTDS_currentContext->currentMessage = NULL;
		} /* End of if ( RTDS_currentContext->currentMessage != NULL ) */

	/* If SDL state has changed and messages have been saved: reorganise the save queue */
	if ( (RTDS_currentContext->sdlState != RTDS_sdlStatePrev) && (RTDS_currentContext->writeSaveQueue != NULL) )
		{
		RTDS_MessageHeader	*message;
		/* Let's get to the end of the save queue */
		for (message=RTDS_currentContext->writeSaveQueue ; message->next != NULL ; message = message->next);
		message->next = RTDS_currentContext->readSaveQueue;
		RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
		RTDS_currentContext->writeSaveQueue = NULL;
		}

	return;
	}



/* **************************************************************** *
 *	RTDS_GetSystemTime
 * **************************************************************** *
 * As its name states...
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 System tick count value
 * **************************************************************** */

portTickType  RTDS_GetSystemTime(void)
	{
	return xTaskGetTickCount();
	}




/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */

#ifdef RTDS_SIMULATOR
unsigned char 				*RTDS_globalMessageUniqueIdPool=NULL;
RTDS_GlobalTraceInfo	RTDS_globalTraceEntry={RTDS_systemError,NULL,0};
char									*RTDS_globalPrintableParameters=NULL;


/* **************************************************************** *
 *	RTDS_GetMessageUniqueId
 * **************************************************************** *
 * Gets a message unique id for the simulator
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 - the message unique id (minimum is 1)
 * **************************************************************** */

unsigned long RTDS_GetMessageUniqueId(void)
	{
	unsigned char *index;
	long			uniqueByteId;
	long			uniqueBitId;

	index=RTDS_globalMessageUniqueIdPool;

	RTDS_CRITICAL_SECTION_START;
	for (uniqueByteId=0;uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE;uniqueByteId++)
		{
		if (*index != 0xFF)
			{
			for (uniqueBitId=0;uniqueBitId<8;uniqueBitId++)
				{
				if ( ( (1<<uniqueBitId) & *index) == 0 )
					{
					*index = *index | (1<<uniqueBitId) ;
					RTDS_CRITICAL_SECTION_STOP;
					return 8*uniqueByteId+uniqueBitId+1;
					}
				}
			}
		index++;
		}
	RTDS_CRITICAL_SECTION_STOP;

	/* All bits are set... No more message unique id */
	RTDS_SYSTEM_ERROR(RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID);
	return 0;
	}




/* **************************************************************** *
 *	RTDS_ReleaseMessageUniqueId
 * **************************************************************** *
 * Make a message unique id available from the pool
 * **************************************************************** *
 * Parameters:
 *	 message unique id
 * Returns:
 *	 nothing
 * **************************************************************** */

void RTDS_ReleaseMessageUniqueId(unsigned long messageUniqueId)
	{
	unsigned char *index;

	if (messageUniqueId == 0) /* probably a timer */
		return;
	messageUniqueId -= 1;
	index = RTDS_globalMessageUniqueIdPool;
	index += (unsigned char)(messageUniqueId/8);
	RTDS_CRITICAL_SECTION_START;
	(*index) = (*index) & ~(1<<messageUniqueId%8);
	RTDS_CRITICAL_SECTION_STOP;
	}



/* **************************************************************** *
 *	RTDS_MemAlloc
 * **************************************************************** *
 * Allocates memory
 * **************************************************************** *
 * Parameters:
 *	 the size of the memory space to allocate
 * Returns:
 *	 a void pointer on the allocated memory space
 * Note:
 *	 This function has been done in the first place to be called
 *	 from the simulator since malloc can not be called directly.
 * **************************************************************** */

void *RTDS_MemAlloc(int size)
	{
	return (void *)malloc(size);
	}



/* **************************************************************** *
 *	RTDS_DummyTraceFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function and reads the RTDS_globalTraceEntry variable to see
 * what happened
 * **************************************************************** *
 * Parameters:
 *	 None
 * Returns:
 *	 nothing
 * **************************************************************** */

void RTDS_DummyTraceFunction(void)
	{
	}

    
#if defined( RTDS_FORMAT_TRACE ) || defined( RTDS_SIMULATOR )
    /* **************************************************************** *
     *    RTDS_GetProcessInfoFromSdlInstanceId
     * **************************************************************** *
     * Returns the current RTDS_GlobalProcessInfo
     * **************************************************************** *
     * Parameters:
     *     RTDS_SdlInstanceId *
     * Returns:
     *     RTDS_GlobalProcessInfo *
     * **************************************************************** */
    RTDS_GlobalProcessInfo * RTDS_GetProcessInfoFromSdlInstanceId( RTDS_SdlInstanceId * sdlInstanceId )
        {
        RTDS_GlobalProcessInfo * processInfo;

        if ( RTDS_globalProcessInfo == NULL )
            {
            return RTDS_globalProcessInfo;
            }
        for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
            {
            if ( processInfo->mySdlInstanceId == sdlInstanceId )
                {
                return processInfo;
                }
            }

        return NULL; /* If not found */
        }
#endif


/* **************************************************************** *
 *    RTDS_GetProcessInfoFromProcessNum
 * **************************************************************** *
 * Returns the RTDS_GlobalProcessInfo of the corresponding process number
 * **************************************************************** *
 * Parameters:
 *     int sdlProcessNumber
 * Returns:
 *     RTDS_GlobalProcessInfo *
 * Use:
 *    presently used by RTDS_SYSTEM_ERROR macro
 * **************************************************************** */
#if defined( RTDS_FORMAT_TRACE ) || defined( RTDS_SIMULATOR )
RTDS_GlobalProcessInfo * RTDS_GetProcessInfoFromProcessNum( int sdlProcessNumber )
    {
    RTDS_GlobalProcessInfo * processInfo;
    
    if ( RTDS_globalProcessInfo == NULL )
        {
        return RTDS_globalProcessInfo;
        }

    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->sdlProcessNumber == sdlProcessNumber )
            {
            return processInfo;
            }
        }
    return NULL; /* If not found */
    }
#endif


/* **************************************************************** *
 *    RTDS_SimulatorMsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue without waiting for acknoledge
 * if using dynamictrace
 * **************************************************************** *
 * Parameters:
 *         - messageNumber representing a message name
 *         - dataLength length of data pointed by pData
 *         - pData    pointer on data sent with the message
 *         - receiver message receiver queue address
 *         - sender     message sender queue address
 * Returns:
 *     Nothing
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) && defined( RTDS_SOCKET_PORT )
    void RTDS_SimulatorMsgQueueSend(
            long                    messageNumber,
            long                    dataLength,
            unsigned char           *pData,
            RTDS_SdlInstanceId      *receiver,
            RTDS_SdlInstanceId      *sender,
            RTDS_GlobalProcessInfo	*RTDS_currentContext )
        {
        RTDS_MessageHeader *RTDS_messageToSend;
        
        RTDS_messageToSend = (RTDS_MessageHeader *)RTDS_MALLOC(sizeof(RTDS_MessageHeader));
        RTDS_messageToSend->messageNumber = messageNumber;
        RTDS_messageToSend->timerUniqueId = 0;
        RTDS_messageToSend->sender = sender;
        RTDS_messageToSend->receiver = receiver;
        RTDS_messageToSend->dataLength = dataLength;
        RTDS_messageToSend->pData = pData;
        RTDS_messageToSend->next = NULL;
        
        #ifdef RTDS_SIMULATOR
            RTDS_CRITICAL_TRACE_SECTION_START
            RTDS_messageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
            RTDS_messageDataToString( &RTDS_globalPrintableParameters, messageNumber, dataLength, ( void * )pData, RTDS_PARAM_CODEC_MAX_DEPTH );
            RTDS_SIMULATOR_TRACE( RTDS_messageSent, RTDS_messageToSend, receiver, RTDS_currentContext );
            RTDS_FREE( RTDS_globalPrintableParameters );
            RTDS_globalPrintableParameters = NULL;
            RTDS_CRITICAL_TRACE_SECTION_STOP
        #endif

        if ( xQueueSend(receiver->queueId,(void *)(&RTDS_messageToSend), portMAX_DELAY) != pdPASS  )
            RTDS_SYSTEM_ERROR(RTDS_ERROR_MSG_QUEUE_SEND);
        }
#endif

        
/* **************************************************************** *
 *    RTDS_IsSemaphoreAvailable
 * **************************************************************** *
 * Returns the semaphore state
 * **************************************************************** *
 * Parameters:
 *     RTDS_SemaphoreId semaphoreId
 * Returns:
 *     int: -1 if function not available
 *                 0 if semaphore not available
 *                 1 if semaphore available
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
long RTDS_IsSemaphoreAvailable( RTDS_SemaphoreId semaphoreId )
    {
    int stillAvailable = -1;
    if ( xSemaphoreTake(semaphoreId, 0) != pdTRUE )
        {
        stillAvailable = 0;
        }
    else
        {
        xSemaphoreGive( semaphoreId );
        stillAvailable = 1;
        }
    return stillAvailable;
    }
#endif



#endif

