/***********************************************************************************
 *                  PragmaDev RTDS FreeRTOS Windows Simulator integration
 ***********************************************************************************
 *                             SOCKET COMMUNICATION FUNCTIONS
 *----------------------------------------------------------------------------------
 * The purpose of these functions is to connect to the host environment through
 * a socket.
 * The information sent through the socket is used
 *     - from the target to the host: to trace SDL events
 *     - from host to target: to stop the target program and send messages to the
 *         system
 * ---------------------------------------------------------------------------------
 *                    + RTDS_InitSocket
 *                    + RTDS_SendSocket
 *                    + RTDS_ReadSocket
 *                    + RTDS_CloseSocket
 ***********************************************************************************/

/* RTDS kernel includes */
#include "RTDS_OS.h"
#include "RTDS_MACRO.h"
#include "RTDS_Error.h"

/* TCP Client include*/
#include "RTDS_TCP_Client.h"

#ifdef RTDS_SOCKET_PORT

/* INCLUDES*/
/* Std include */
#include <stdlib.h>
#include <stdio.h>

#ifdef __CYGWIN__
    #include <unistd.h>
    #include <sys/socket.h>
    #include <netinet/in.h>
    #include <arpa/inet.h>
    #include <netdb.h>
#else
    #include <windows.h>
    #include <Winsock2.h>
    #include <Ws2tcpip.h>
    #include <wincon.h>
    #include <stdlib.h>
    #include <stdio.h>
    #include <time.h>
    #include <ctype.h>
#endif




/***********************************************************************************************************
 *                                 FUNCTION  RTDS_InitSocket(...)
 *----------------------------------------------------------------------------------------------------------
 *    Author    : Pragmadev S.P
 *    Date        : 06/06/02
 *    Purpose : start socket
 ***********************************************************************************************************/
RTDS_SOCKET_ID_TYPE RTDS_InitSocket( char * hostNameString, int portNum )
    {
    int sd;
    int rc = -1;
    struct sockaddr_in localAddr;
    struct sockaddr_in servAddr;
    int clientSocketId = 0;
    
#ifndef __CYGWIN__
    WORD wVersionRequested;
    WSADATA wsaData;
    int err;
    wVersionRequested = MAKEWORD( 2, 2 );
    err = WSAStartup( wVersionRequested, &wsaData );
#endif
    
    servAddr.sin_family = AF_INET;
    servAddr.sin_addr.s_addr = inet_addr( hostNameString );  /* Server IP address */
    servAddr.sin_port = htons( portNum );
    
    /* Create socket */
    sd = socket( AF_INET, SOCK_STREAM, 0 );
    
    if( sd < 0 )
        {
        /* Cannot open socket */
        return RTDS_ERROR;
        }
    else
        {
        clientSocketId = sd;
        }
    /* Bind any port number */
    localAddr.sin_family = AF_INET;
    localAddr.sin_addr.s_addr = htonl( INADDR_ANY );
    localAddr.sin_port = htons( 0 );
    
    rc = bind( sd, ( struct sockaddr * ) &localAddr, sizeof( localAddr ) );
    
    if( rc < 0 )
        {
        /* Cannot bind port */
        return RTDS_ERROR;
        }
    
    rc = connect( sd, ( struct sockaddr * ) &servAddr, sizeof( servAddr ) );
    
    if( rc != 0 )
        {
        /* Could not connect to server */
        return RTDS_ERROR;
        }
    return clientSocketId;
    }

/******************************************************************************************
 *                                 FUNCTION  RTDS_CloseSocket(...)
 *-----------------------------------------------------------------------------------------
 *    Purpose  :
 *
 *    return status RTDS_OK / RTDS_ERROR
 *
 ******************************************************************************************/
int RTDS_CloseSocket( RTDS_SOCKET_ID_TYPE socketId )
    {
    /* Since no way to stop task if not self : this function    doesn't attempt to kill thread */
#ifdef __CYGWIN__
    close( socketId );
#else
    closesocket( socketId );
#endif
    
    return RTDS_OK;
    }

/******************************************************************************************
 *                                 FUNCTION  RTDS_ReadSocket(...)
 *-------------------------------------------------------------------
 *    Purpose : return a valid data (ended by END_STRING) or RTDS_ERROR
 *                        if socket connection and reading probleme.
 *
 *    Release : This version of function read and process char by char
 *                and does not use any buffer to read a certain
 *                        amount of data before processing.
 *
 *    socketDescriptor : socket ID
 *    data                         : read buffer: pointer on the value to return
 *    delimiterChar      : delimiter char
 *
 *
 *    return                     : currentSize or -1 if failing
 ******************************************************************************************/
int RTDS_ReadSocket( RTDS_SOCKET_ID_TYPE socketId, char ** data, int * dataSize, char delimiterChar, char escapeChar )
    {
    static int n;
    int offset = 0;
    char * tmpReadBuffer;
    int blockSize = 1024;
    int blockFreeSpace;     /* Used to extend memory chunk if needed */
    int nbBlock = 1;        /* Used to count the number of block allocated for the reda buffer */
    char tmpchar;
    int escapeFlag = 0;     /* indicate that the previous received char was the escapeChar*/
    
    /* Init the block free space to the initial size */
    blockFreeSpace = blockSize;
    tmpReadBuffer = ( char * )RTDS_MALLOC( blockSize );
    *data = NULL;
    
    while( 1 )
        {
        n = recv( socketId, &tmpchar, sizeof( char ), 0 );
        if( n < 0 )
            {
            /* Cannot receive data */
            return -1;
            }
        else if( n == 0 )
            {
            /* Connection closed by server */
            RTDS_CloseSocket( socketId );
            return -1;
            }
        else if( n == 1 )
            {
            if( escapeFlag == 1 )
                {
                if ( tmpchar == delimiterChar )
                    {
                    /* escapeChar was followed by delimiterChar */
                    *dataSize = offset + 2;
                    tmpReadBuffer[ offset ] = tmpchar;
                    *data = tmpReadBuffer;
                    break;
                    }
                else
                    {
                    /* escapeChar was not followed by delimiterChar: wait for escapeChar */
                    escapeFlag = 0;
                    }
                }
                else
                    {
                    /* waiting for escapeChar */
                    if ( tmpchar == escapeChar )
                        {
                        escapeFlag = 1;
                        }
                    }
            /* Add char and increment offset */
            tmpReadBuffer[ offset ] = tmpchar;
            offset++;
            
            if ( !blockFreeSpace )
                {
                nbBlock++;
                tmpReadBuffer = (char *)realloc(tmpReadBuffer, nbBlock * blockSize);
                if ( tmpReadBuffer = NULL )
                    {
                    RTDS_SYSTEM_ERROR( RTDS_ERROR_EXTEND_MEMORY_ALLOC );
                    }
                blockFreeSpace = blockSize;
                }
            else
                {
                blockFreeSpace--;
                }
            }/* end of else if n==1 */
        } /* end of while */
    
    return offset;
    }

/******************************************************************************************
 *                                 FUNCTION  RTDS_SendSocket(...)
 *-------------------------------------------------------------------
 *    Purpose : Send data over the socket
 *
 *    socketId                : socket ID
 *    commandString     : pointer on data to send
 *    size                        : size to send
 *
 *    return                    : RTDS_OK or RTDS_ERROR
 ******************************************************************************************/
int RTDS_SendSocket( RTDS_SOCKET_ID_TYPE socketId, char * commandString, int size )
    {
    int stat = -1;
    
    /* send msg */
    stat = send( socketId, commandString, ( size_t )size, 0 );
    
    if ( stat < 1 )
        {
        stat = RTDS_ERROR;
        }
    else
        {
        stat = RTDS_OK;
        }
    return stat;
    }

#endif /* RTDS_SOCKET_PORT */


#ifdef RTDS_FREERTOS_WINDOWS_SIMULATOR
    /*
     * SEMAPHORE PROTOTYPE:
     * -------------------
     * Variable declared as global to the all the program
     */
    RTDS_CRITICAL_SECTION_PROTO;
    RTDS_START_SYNCHRO_PROTO;
    RTDS_CRITICAL_TRACE_SECTION_PROTO;
    RTDS_SOCKET_ACCESS_PROTO;
    RTDS_DTRACE_ACKNOWLEDGE_PROTO;
    
    #if defined( RTDS_SOCKET_PORT )
        extern int globalClientSocketId;
    #endif

    
/********************************************************************
 *                     FUNCTION  RTDS_SemMCreate
 *-------------------------------------------------------------------
 * Create a mutex semaphore
 *
 * PARAMETERS
 *        None
 * RETURN
 *        SemaphoreId : RTDS_WindowsSemaphoreId
 *********************************************************************/
RTDS_SemaphoreId RTDS_SemMCreate( void )
    {
    RTDS_WindowsSemaphoreId SemId;
    
    SemId = ( RTDS_SemaphoreId )RTDS_MALLOC( sizeof( RTDS_SemaphoreInfo ) );
    SemId->semId = CreateMutex( NULL, FALSE, NULL );
    if ( SemId->semId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_WINDOWS_CREATE_MUTEX_SEMAPHORE );
        }
    else
        {
        SemId->semType = MUTEX;
        return SemId;
        }
    }
    
    
    /********************************************************************
     *                   RTDS_SemBCreate
     *-------------------------------------------------------------------
     * Creates a binary semaphore
     *-------------------------------------------------------------------
     * PARAMETERS:
     *      initialState : int  initial state (0: Empty / 1: Full)
     * RETURN:
     *      SemaphoreId : RTDS_WindowsSemaphoreId
     *********************************************************************/
    RTDS_WindowsSemaphoreId RTDS_SemBCreate( int initial_state )
        {
        RTDS_WindowsSemaphoreId SemId;
        
        SemId = ( RTDS_WindowsSemaphoreId )RTDS_MALLOC( sizeof( RTDS_SemaphoreInfo ) );
        SemId->semId = CreateSemaphore( NULL, initial_state, 1, NULL );
        if ( SemId->semId == NULL )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_WINDOWS_CREATE_BINARY_SEMAPHORE );
            }
        else
            {
            SemId->semType = BINARY;
            return SemId;
            }
        }

    /********************************************************************
     *                       RTDS_SemDelete
     *-------------------------------------------------------------------
     * The purpose of this function is to create binary semaphore
     * (Works for mutex and regular semaphore)
     *-------------------------------------------------------------------
     * PARAMETERS:
     *      semId : RTDS_WindowsSemaphoreId
     * RETURN:
     *      Nothing
     *********************************************************************/
    void RTDS_SemDelete( RTDS_WindowsSemaphoreId semId )
        {
        CloseHandle( semId->semId );
        RTDS_FREE( semId );
        }

    /********************************************************************
     *                        RTDS_SemGive
     *-------------------------------------------------------------------
     * The purpose of this function is to release semaphore
     * (Works for mutex and regular semaphore)
     *-------------------------------------------------------------------
     * Assum that the semaphore is a counting (regular semaphore) and try 
     * to increment the counter if operation fails try Mutex command 
     * => if both returns FALSE then return RTDS_ERROR.
     *-------------------------------------------------------------------
     * PARAMETERS:
     *        semId : RTDS_WindowsSemaphoreId        semaphore ID
     *********************************************************************/
    void RTDS_SemGive( RTDS_WindowsSemaphoreId semId )
        {
        /* For mutex or regular semaphore */
        if ( ( semId->semType == COUNTING ) || ( semId->semType == BINARY ) )
            {
            if ( ReleaseSemaphore( semId->semId, 1, NULL ) == 0 )
                {
                RTDS_SYSTEM_ERROR( RTDS_ERROR_WINDOWS_SEMAPHORE_GIVE );
                }
            }
        else
            {
            if ( ReleaseMutex( semId->semId ) == 0 )
                {
                RTDS_SYSTEM_ERROR( RTDS_ERROR_WINDOWS_SEMAPHORE_GIVE );
                }
            }
        }

    /********************************************************************
     *                      RTDS_SemTake
     *-------------------------------------------------------------------
     * Takes a mutex or a regular semaphore
     *-------------------------------------------------------------------
     * PARAMETERS:
     *        semId: RTDS_WindowsSemaphoreId    semaphore ID
     *
     * RETURN:
     *        Semaphore status RTDS_OK or RTDS_TIMEOUT
     *********************************************************************/
    int RTDS_SemTake( RTDS_WindowsSemaphoreId semId, long TIME_TO_WAIT )
        {
        int status;
        
        status = WaitForSingleObject( semId->semId, TIME_TO_WAIT );
        
        if ( status == WAIT_TIMEOUT )
            {
            return RTDS_TIMEOUT;
            }
        else if ( status == WAIT_OBJECT_0 )
            {
            return RTDS_OK;
            }
        else
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_WINDOWS_SEMAPHORE_TAKE );
            }
        }
        
        
#endif /* RTDS_FREERTOS_WINDOWS_SIMULATOR */

/* Function defined is trace needed */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )

  /* **************************************************************** *
   *    RTDS_SimulatorTrace
   * **************************************************************** *
   * As its name states... Handles trace with the host depending on
   * the type of link available and options set in the profile.
   * **************************************************************** *
   * Parameters:
   *    enum RTDS_EventType  event:                         trace event
   *    void *eventParameter1:                                    parameter 1 of the trace
   *    long eventParameter2:                                     parameter 2 of the trace
   *    RTDS_GlobalProcessInfo *currentContext: calling context
   * Returns:
   *     nothing
   * **************************************************************** */

  void RTDS_SimulatorTrace( enum RTDS_EventType event, void * eventParameter1, long eventParameter2, RTDS_GlobalProcessInfo * currentContext, int waitAck )
      {
  #ifdef RTDS_FORMAT_TRACE
      int formatStatus = RTDS_ERROR;
      char * formattedCmd = NULL;
  #endif /* RTDS_FORMAT_TRACE */
      
      unsigned long sysTime = 0;
      sysTime = RTDS_GetSystemTime();
      
      if ( waitAck == RTDS_DTRACE_ACK_WAIT )
          {
          RTDS_CRITICAL_TRACE_SECTION_START;
          }
      
      /* Format trace */
      RTDS_globalTraceEntry.event = event;
      RTDS_globalTraceEntry.eventParameter1 = ( void * )eventParameter1;
      RTDS_globalTraceEntry.eventParameter2 = ( long )eventParameter2;
      RTDS_globalTraceEntry.currentContext = ( RTDS_GlobalProcessInfo * )currentContext;
  
  #ifdef RTDS_FORMAT_TRACE
      formatStatus = RTDS_FormatTrace( sysTime, RTDS_globalTraceEntry, waitAck, &formattedCmd );
      /* Send trace to RTDS or MSC Tracer */
  #if defined( RTDS_SOCKET_PORT )
      if ( formattedCmd != NULL )
          {
          RTDS_SOCKET_ACCESS_TAKE;
          RTDS_SendSocket( globalClientSocketId, formattedCmd, strlen( formattedCmd ) );
          RTDS_SOCKET_ACCESS_GIVE;
          if ( waitAck == RTDS_DTRACE_ACK_WAIT )
              { 
              RTDS_DTRACE_ACKNOWLEDGE_WAIT; 
              }
          }
  #endif
  
  #ifdef RTDS_SIMULATOR
      /* Called for backtrace*/
      RTDS_DummyTraceFunction();
  #endif
  
      RTDS_FREE( formattedCmd );
  
  #else /* not RTDS_FORMAT_TRACE, just RTDS_SIMULATOR*/
      if ( ( event == RTDS_messageSent ) || ( event == RTDS_messageReceived ) )
          {
          RTDS_messageDataToString( &RTDS_globalPrintableParameters, ( ( RTDS_MessageHeader * )eventParameter1 )->messageNumber, ( ( RTDS_MessageHeader * )eventParameter1 )->dataLength, ( void * )( ( ( RTDS_MessageHeader * )eventParameter1 )->pData ), RTDS_PARAM_CODEC_MAX_DEPTH );
          }
      RTDS_DummyTraceFunction();
      if ( ( event == RTDS_messageSent ) || ( event == RTDS_messageReceived ) )
          {
          RTDS_FREE( RTDS_globalPrintableParameters );
          RTDS_globalPrintableParameters = NULL;
          }
  #endif
      
      if ( waitAck == RTDS_DTRACE_ACK_WAIT )
          {
          RTDS_CRITICAL_TRACE_SECTION_STOP;
          }
      }

#endif /* defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACER) */

