/* PragmaDev RTDS OSE 52 integration */
#if defined( RTDS_SOCKET_IP_ADDRESS )

/* OSE includes */
#include "sys/time.h"
#include "sys/types.h"
#include "sys/socket.h"
#include "netinet/in.h"
#include "netinet/tcp.h"
#include "arpa/inet.h"
#include "unistd.h"
#include "dbgprintf.h"


/* RTDS includes */
#include "RTDS_Trace.h"
#include "string.h"

/* Debug */
#include <stdio.h>

#ifdef RTDS_CMD_INTERPRETER
#include "RTDS_CmdInterpreter.h"
#endif

#ifdef RTDS_FORMAT_TRACE
#include "RTDS_FormatTrace.h"
#endif

RTDS_SocketId socketDescriptor;
RTDS_DTRACE_ACKNOWLEDGE_DECL;
RTDS_SOCKET_ACCESS_DECL;
RTDS_CRITICAL_TRACE_SECTION_DECL;

/* **************************************************************** *
 * RTDS_InitSocket
 * **************************************************************** *
 * Initialize socket connection to host to send trace information
 * **************************************************************** *
 * Parameters:
 *      - None
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_InitSocket( void )
    {
    struct sockaddr_in faddr;  /* Foreign address (destination) */
    char * ipAddress;
    int attempt = 0;
    
    socketDescriptor = socket( AF_INET, SOCK_STREAM, 0 );
    
    if ( socketDescriptor == -1 )
        {
        dbgprintf( "RTDS_InitSocket: invalid socket descriptor\n" );
        return;
        }
    
    dbgprintf( "RTDS_InitSocket: socket descriptor: %d\n", socketDescriptor );
    
    ipAddress = GET_HOSTNAME_STRING( RTDS_SOCKET_IP_ADDRESS );
    
    dbgprintf( "RTDS_InitSocket: ip address: %s\n", ipAddress );
    
    memset( (void*)&faddr, 0, sizeof( faddr ) );
    faddr.sin_len = sizeof( struct sockaddr_in );
    faddr.sin_family = AF_INET;
    faddr.sin_addr.s_addr = inet_addr( ipAddress );
    faddr.sin_port = htons( RTDS_SOCKET_PORT );
    
    if( faddr.sin_addr.s_addr == INADDR_NONE )
        {
        dbgprintf( "RTDS_InitSocket: invalid IP address\n" );
        close( socketDescriptor );
        socketDescriptor = -1;
        return;
        }
    
    while ( connect(socketDescriptor, ( struct sockaddr * )&faddr, sizeof( faddr ) ) != 0 )
        {
        if( attempt > 5 )
            {
            dbgprintf( "RTDS_SendSocket: Try to connect more than 5 times !\n" );
            dbgprintf( "RTDS_SendSocket: Check your configuration. Cannot connect to the debugger !\n" );
            close( socketDescriptor );
            socketDescriptor = -1;
            return;
            }
        dbgprintf( "RTDS_InitSocket: connect *(inet_errno()): %d\n", *( inet_errno() ) );
        dbgprintf( "RTDS_SendSocket: Try to connect again....Please wait\n" );
        attempt++;
        }
    
    dbgprintf( "RTDS_SendSocket: Connection OK!\n" );
    }

/* **************************************************************** *
 * RTDS_SendSocket
 * **************************************************************** *
 * Sends textual trace information as is through the socket
 * connection to host
 * **************************************************************** *
 * Parameters:
 *      - char * buffer: textual information to send
 *      - long dataLength: length of the textual info to send
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_SendSocket( char * buffer, long dataLength )
    {
    int status;
    
    if ( socketDescriptor == -1 )
        {
        dbgprintf( "RTDS_SendSocket: Invalid socket descriptor\n" );
        return;
        }
    status = inet_send( socketDescriptor, ( void * )buffer, ( int )dataLength, 0 ) ;
    if ( status != dataLength )
        {
        printf( "status=%d dataLength=%d strlen(buffer)=%d sizeof(buffer)=%d\n", status, dataLength, strlen( buffer ), sizeof( buffer ) );
        error2( RTDS_ERROR_SOCKET_SEND_ERROR_WRONG_SENT_DATA, ( OSERRCODE )"Check error code in $(RTDS_HOME)/share/ccg/ose52/RTDS_Error.h" );
        }
    }

/* **************************************************************** *
 * RTDS_CloseSocket
 * **************************************************************** *
 * Closes socket connection to host
 * **************************************************************** *
 * Parameters:
 *      - none
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_CloseSocket( void )
    {
    if ( socketDescriptor != -1 )
        {
        close( socketDescriptor );
        }
    }

#ifdef RTDS_CMD_INTERPRETER
/* *****************************************************************************************
 * RTDS_ReadSocket
 * *****************************************************************************************
 * Purpose : reads a socket until it is terminated by escapeChar followed by delimiterChar.
 * Note: The returned buffer is allocated in the function.
 * *****************************************************************************************
 * Inputs:
 *      - RTDS_SOCKET_ID_TYPE localSocketDescriptor,	Socket identifier
 *      - char **pData, Buffer to return
 *      - int *pDataSize, Size of data in the buffer
 *      - char delimiterChar, Delimiter character
 *      - char escapeChar Escape character
 * Returns:
 *      - int RTDS_ERROR or RTDS_OK
 * ***************************************************************************************** */
int RTDS_ReadSocket( char ** pData, int * pDataSize, char delimiterChar, char escapeChar )
    {
    int n;
    char * tmpReadBuffer;
    int blockSize = 1024;   /* Default block size */
    int nbBlock = 1;        /* Used to count the number of block allocated for the buffer */
    char tmpchar;
    int escapeFlag = 0;       /* indicate that the previous received char was the escapeChar*/
    
    /* Allocate a single block size of memory space */
    *pData = ( char * )RTDS_MALLOC( blockSize );
    memset( *pData, 0, blockSize );
    *pDataSize = 0;
    
    /* Loop until a command end is reached */
    while( 1 )
        {
        /* reads caracters one by one */
        n = recv ( socketDescriptor, ( void * )&tmpchar, 1, 0 );
        if ( n <= 0 )
            {
            return RTDS_ERROR;
            }
        else
            {
            /* Add char and increment *pDataSize */
            ( *pData )[ ( *pDataSize ) ] = tmpchar;
            ( *pDataSize )++;
            /* The end of the command is reached when the escapeChar is followed by the delimiterChar */
            if( escapeFlag == 1 )
                {
                /* escapeChar was followed by delimiterChar: end of command */
                if ( tmpchar == delimiterChar )
                    {
                    break;
                    }
                /* escapeChar was not followed by delimiterChar: wait for escapeChar */
                else
                    {
                    escapeFlag = 0;
                    }
                }
            /* waiting for escapeChar */
            else if ( tmpchar == escapeChar )
                    {
                    escapeFlag = 1;
                    }
            
            /* Is the allocated block full ? */
            if ( *pDataSize >= nbBlock * blockSize )
                {
                nbBlock++;
                tmpReadBuffer = ( char * )RTDS_MALLOC( nbBlock * blockSize );
                memset( tmpReadBuffer, 0, nbBlock * blockSize );
                memcpy( tmpReadBuffer, *pData, nbBlock * blockSize );
                RTDS_FREE( *pData );
                *pData = tmpReadBuffer;
                }
            }/* end of if n<=0 */
        } /* end of while */
    
    return RTDS_OK;
    }

/* **************************************************************** *
 * RTDS_StopProgram
 * **************************************************************** *
 * Dummy function to stop program through the socket
 * **************************************************************** *
 * Parameters:
 *      - None
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_StopProgram( void )
    {
    }

/* **************************************************************** *
 * RTDS_GetProcessInfoFromProcessId
 * **************************************************************** *
 * Gets the current context of the process identified by its pid
 * **************************************************************** *
 * Parameters:
 *      - RTDS_SdlInstanceId
 * Returns:
 *      - RTDS_GlobalProcessInfo * currentContext
 * **************************************************************** */
RTDS_GlobalProcessInfo * RTDS_GetProcessInfoFromProcessId( RTDS_SdlInstanceId * sdlInstanceId )
    {
    RTDS_GlobalProcessInfo * processInfo;
    
    if ( RTDS_globalProcessInfo == NULL )
        {
        return RTDS_globalProcessInfo;
        }
    for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->mySdlInstanceId == sdlInstanceId )
           {
           return processInfo;
           }
        }
    return NULL; /* If not found */
    }

/* **************************************************************** *
 * RTDS_GetProcessInfoFromProcessNum
 * **************************************************************** *
 * Returns the RTDS_GlobalProcessInfo of the corresponding process number
 * **************************************************************** *
 * Parameters:
 *      - int processNumber
 * Returns:
 *      - RTDS_GlobalProcessInfo *
 * Use:
 *      - presently used by RTDS_SYSTEM_ERROR macro
 * **************************************************************** */
RTDS_GlobalProcessInfo * RTDS_GetProcessInfoFromProcessNum( int processNumber )
    {
    RTDS_GlobalProcessInfo * processInfo;
    
    if ( RTDS_globalProcessInfo == NULL )
        {
        return RTDS_globalProcessInfo;
        }
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->sdlProcessNumber == processNumber )
            {
            return processInfo;
            }
        }
    return NULL; /* If not found */
    }

/* **************************************************************** *
 * RTDS_SocketProcess
 * **************************************************************** *
 * OSE process to handle incoming events from the socket
 * **************************************************************** *
 * Parameters:
 *      - None
 * Returns:
 *      - Nothing
 * **************************************************************** */
RTDS_TASK_ENTRY_POINT( RTDS_SocketProcess )
    {
    char * pReceivedData;
    int receivedDataSize;   /* Data size to read from socket */
    enum RTDS_CmdType commandId;
    RTDS_UnionCommandParam * pCommandParam;
    RTDS_CmdSendMsg2ProcId * paramSendMsg2pId = NULL;
    RTDS_CmdSendMsg2ProcNumber * paramSendMsg2pNumber = NULL;
    
    while( RTDS_ReadSocket( &pReceivedData, &receivedDataSize, RTDS_CMD_DELIMITERCHAR, RTDS_ESCAPECHAR ) != RTDS_ERROR )
        {
        if ( RTDS_InterpretCommand( pReceivedData, &commandId, &pCommandParam ) == RTDS_OK )
            {
            /* Process command */
            switch( commandId )
                {
                case ACK_CMDID:
                    RTDS_DTRACE_ACKNOWLEDGE_RECEIVE;
                    break;
                
                case STOP_CMDID:
                    RTDS_StopProgram();
                    break;
                
                case MSG2PROCID:
                    /* pCommandParam is a valid RTDS_CmdSendMsg2ProcId struct */
                    paramSendMsg2pId = ( RTDS_CmdSendMsg2ProcId * )pCommandParam;
                    /* With OSE integration pid and queueid are the same */
                    RTDS_SimulatorMsgQueueSend(
                    paramSendMsg2pId->messageNumber,
                    paramSendMsg2pId->dataLength,
                    paramSendMsg2pId->pData,
                    paramSendMsg2pId->receiver,
                    paramSendMsg2pId->sender,
                    RTDS_GetProcessInfoFromProcessId( paramSendMsg2pId->sender ) );
                    break;
                
                case MSG2PROCNUM:
                    /* pCommandParam is a valid RTDS_CmdSendMsg2ProcNumber struct */
                    paramSendMsg2pNumber = ( RTDS_CmdSendMsg2ProcNumber * )pCommandParam;
                    RTDS_SimulatorMsgQueueSend(
                    paramSendMsg2pNumber->messageNumber,
                    paramSendMsg2pNumber->dataLength,
                    paramSendMsg2pNumber->pData,
                    RTDS_GetProcessQueueId( paramSendMsg2pNumber->receiver ),
                    RTDS_GetProcessQueueId( paramSendMsg2pNumber->sender ),
                    RTDS_GetProcessInfoFromProcessNum( paramSendMsg2pNumber->sender ) );
                    break;
                
                case TRACE_CMDID:
                #ifdef RTDS_FORMAT_TRACE
                    RTDS_ConfigureTrace( ( RTDS_CmdTraceConfig * )pCommandParam );
                #endif
                    break;
                
                default:
                    break;
                } /* end of switch(commandId) */
            } /* end of if (RTDS_InterpretCommand...) */
        else
            {
            return;
            }
        RTDS_FREE( pReceivedData );
        } /* end of while( RTDS_ReadSocket...) */
    } /* end of RTDS_TASK_ENTRY_POINT(RTDS_SocketProcess) */
    
#endif /* RTDS_CMD_INTERPRETER */
#endif /* RTDS_SOCKET_IP_ADDRESS */

