#include <stdlib.h>
#include <string.h>

#include "RTDS_MACRO.h"
#include "RTDS_OS.h"

#include "RTDS_InternalConstants.h"

#include "RTDS_BinarySemaphoreProcess.h"
#include "RTDS_CountingSemaphoreProcess.h"
#include "RTDS_MutexSemaphoreProcess.h"


/*
** GLOBAL VARIABLES:
** =================
*/

/* RTDS_globalTimerListFirst: head of list of all timers active in the system */
static RTDS_TimerState * RTDS_globalTimerListFirst = NULL;

/* RTDS_globalTimerListCurrent: first element in list of all timers that has not timed-out */
static RTDS_TimerState * RTDS_globalTimerListCurrent = NULL;

/* Pool for message unique identifiers (only in debug) */
#ifdef RTDS_SIMULATOR
void * RTDS_globalMessageUniqueIdPool = NULL;
#endif

/* Global variables for trace entry and encoded message parameters when needed */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE )
RTDS_GlobalTraceInfo RTDS_globalTraceEntry = { RTDS_systemError , NULL , 0 };
char *RTDS_globalPrintableParameters = NULL;
#endif


/*
** FUNCTION RTDS_MsgQCreate:
** -------------------------
** Supposed to create and return a new queue for a scheduler process. Since
** no RTOS is present, this doesn't create anything and returns NULL.
*/

RTDS_QCB * RTDS_MsgQCreate( void )
  {
  return (RTDS_QCB*)NULL;
  }


  
/*
** FUNCTION RTDS_SystemTick:
** -------------------------
** Called whenever the time increments. Should be called by an interrupt.
*/

static unsigned long RTDS_globalSystemTime = 0;
void RTDS_SystemTick(void)
  {
  /* Increment current time */
  RTDS_globalSystemTime++;
  /* If there is a current timer */
  if ( RTDS_globalTimerListCurrent != NULL )
    {
    /* Decrement its delay */
    RTDS_globalTimerListCurrent->timeoutDelay --;
    /* Make timer time-out if needed, along with all the ones with the same time-out time */
    while ( RTDS_globalTimerListCurrent->timeoutDelay == 0 )
      {
      RTDS_globalTimerListCurrent = RTDS_globalTimerListCurrent->next;
      if ( RTDS_globalTimerListCurrent == NULL ) break;
      }
    }
  }
  
  
/*
** FUNCTION RTDS_incomingSdlEvent:
** -------------------------------
** Default implementation for function handling external events.
*/

#ifdef RTDS_HANDLE_EXTERNAL_EVENTS
short RTDS_incomingSdlEvent(RTDS_MessageHeader * message, long timeLeft)
  {
  /*
  ** The default implementation is to increment the system time automatically.
  ** This should *NOT* be done in production systems, where the system time is
  ** supposed to be incremented via an interrupt, and not handled here. This
  ** default implementation is just to ensure that systems running on a host
  ** work correctly, especially timers...
  */
  RTDS_SystemTick();
  return 0;
  }
#endif


/*
** FUNCTION RTDS_Wait4ExternalMessage:
** -----------------------------------
** Waits for an external message; called each time the scheduler has
** exhausted all its internal messages.
** An external message is either a message coming from the environment,
** usually triggered by an interrupt, or a message due to a timer time-out.
** Parameters:
** - message: pointer on a pre-allocated message
** Returns: RTDS_OK if the message has been correctly read, RTDS_ERROR if
** the reading fails for any reason.
*/

int RTDS_Wait4ExternalMessage(RTDS_MessageHeader * message)
  {
  RTDS_TimerState * timedOutTimer = NULL;
  long              timeLeft = -1;
  
  /* Wait for an event until one happens */
  for ( ; ; )
    {
    /* We'll access the global timer list: make sure it won't get updated */
    RTDS_DISABLE_INTERRUPTS;
    /* First considered external messages: timers => discard first cancelled ones */
    while ( ( RTDS_globalTimerListFirst != NULL ) &&
            ( RTDS_globalTimerListFirst->timeoutDelay == 0 ) &&
            ( RTDS_globalTimerListFirst->state == RTDS_TIMER_CANCELLED ) )
      {
      timedOutTimer = RTDS_globalTimerListFirst;
      RTDS_globalTimerListFirst = RTDS_globalTimerListFirst->next;
      RTDS_FREE(timedOutTimer)
      }
    /* If there is a timed-out timer */
    if ( ( RTDS_globalTimerListFirst != NULL ) && ( RTDS_globalTimerListFirst->timeoutDelay == 0 ) )
      {
      /* Get timed out timer */
      timedOutTimer = RTDS_globalTimerListFirst;
      RTDS_globalTimerListFirst = RTDS_globalTimerListFirst->next;
      /* Build message for timer */
      message->messageNumber = timedOutTimer->timerNumber;
      message->timerUniqueId = timedOutTimer->timerUniqueId;
      message->sender = timedOutTimer->receiverId;
      message->receiver = timedOutTimer->receiverId;
      message->dataLength = 0;
      message->pData = NULL;
      message->next = NULL;
      /* Free timer descriptor */
      RTDS_FREE(timedOutTimer);
      /* Over */
      RTDS_ENABLE_INTERRUPTS;
      return RTDS_OK;
      }
    /* If there is no timed-out timer, call function getting next external SDL message */
    if ( RTDS_globalTimerListCurrent != NULL )
      timeLeft = RTDS_globalTimerListCurrent->timeoutDelay;
    RTDS_ENABLE_INTERRUPTS;
    /* If function actually gets a message, over */
    if ( RTDS_incomingSdlEvent(message, timeLeft) )
      return RTDS_OK;
    }
  return RTDS_ERROR;
  }


/*
** FUNCTION RTDS_DummyTraceFunction:
** -----------------------------
** Used for tracing: The simulator sets a breakpoint on this function and reads
** the RTDS_globalTraceEntry global variable to see what happened.
*/

#ifdef RTDS_SIMULATOR
void RTDS_DummyTraceFunction( void )
  {
#ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
  RTDS_TraceAdd();
#endif
  }
#endif /* RTDS_SIMULATOR */
  
  
/*
** FUNCTION RTDS_SimulatorTrace:
** -----------------------------
** Handles trace with the host depending on the type of link available and options set in the profile.
**
** PARAMETERS:
**  event :           Trace event                
**  eventParameter1 : Parameter 1 of the trace   
**  eventParameter2 : Parameter 2 of the trace   
**  currentContext :  Calling context            
**  waitAck :         If true, an acknoledgement should be expected from the simulator we communicate with.
*/

#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
void RTDS_SimulatorTrace(
  enum RTDS_EventType       event,
  void                    * eventParameter1,
  long                      eventParameter2,
  RTDS_GlobalProcessInfo  * currentContext,
  int                       waitAck
)
  {
#if defined(RTDS_FORMAT_TRACE)
  unsigned long   sysTime = 0;
  int             formatStatus = RTDS_ERROR;
  char          * formattedCmd = NULL;
  size_t          command_size;
#endif /* defined(RTDS_FORMAT_TRACE) */

  /* Update global trace information */
  RTDS_globalTraceEntry.event = event;
  RTDS_globalTraceEntry.eventParameter1 = (void*)eventParameter1;
  RTDS_globalTraceEntry.eventParameter2 = (long)eventParameter2;
  RTDS_globalTraceEntry.currentContext = (RTDS_GlobalProcessInfo*)currentContext;

#if defined(RTDS_FORMAT_TRACE)
  formatStatus = RTDS_FormatTrace(RTDS_globalSystemTime, RTDS_globalTraceEntry, waitAck, &formattedCmd);
  if ( formattedCmd != NULL )
    {
#if defined(RTDS_SOCKET_PORT)
    /* Send Trace to the external program: RTDS or MSC tracer */
    command_size = strlen(formattedCmd);
    if ( send(globalClientSocketId, formattedCmd, command_size, 0) != command_size )
      {
      RTDS_SYSTEM_ERROR(RTDS_ERROR_SEND)
      }
    if ( waitAck == RTDS_DTRACE_ACK_WAIT )
      { 
      RTDS_DTRACE_ACKNOWLEDGE_WAIT; 
      }
#endif /* defined(RTDS_SOCKET_PORT) */
    }
#if defined(RTDS_SIMULATOR)
    RTDS_DummyTraceFunction();
#endif /* defined(RTDS_SIMULATOR) */
    RTDS_FREE( formattedCmd );
#elif defined(RTDS_SIMULATOR)
    if ( ( event == RTDS_messageSent ) || ( event == RTDS_messageReceived ) )
      {
      RTDS_messageDataToString(
        &RTDS_globalPrintableParameters,
        ((RTDS_MessageHeader*)eventParameter1)->messageNumber,
        ((RTDS_MessageHeader*)eventParameter1)->dataLength,
        (void *)(((RTDS_MessageHeader*)eventParameter1)->pData),
        RTDS_PARAM_CODEC_MAX_DEPTH
      );
      }
    RTDS_DummyTraceFunction();
    if ( ( event == RTDS_messageSent ) || ( event == RTDS_messageReceived ) )
        {
        RTDS_FREE(RTDS_globalPrintableParameters);
        RTDS_globalPrintableParameters = NULL;
        }
#endif /* defined(RTDS_FORMAT_TRACE) / defined(RTDS_SIMULATOR) */
  }
#endif /* defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACER) */


/*
** FUNCTION RTDS_GetMessageUniqueId:
** -------------------------------------
** Gets a message unique id for the simulator.
*/

#ifdef RTDS_SIMULATOR
unsigned long RTDS_GetMessageUniqueId( void )
  {
  unsigned char *index;
  long uniqueByteId;
  long uniqueBitId;

  index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;

  for ( uniqueByteId = 0 ; uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE ; uniqueByteId++ )
    {
    if ( *index != 0xFF )
      {
      for ( uniqueBitId = 0 ; uniqueBitId < 8 ; uniqueBitId++ )
        {
        if ( ( ( 1 << uniqueBitId ) & *index ) == 0 )
          {
          *index = *index | ( 1 << uniqueBitId );
          RTDS_CRITICAL_SECTION_STOP;
          return (8 * uniqueByteId + uniqueBitId + 1);
          }
        }
      }
    index++;
    }

  /* All bits are set... No more message unique id */
  RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID )
  return 0;
  }
#endif

  
/*
** FUNCTION RTDS_ReleaseMessageUniqueId:
** -------------------------------------
** Make a message unique id available from the pool.
*/

#ifdef RTDS_SIMULATOR
void RTDS_ReleaseMessageUniqueId( unsigned long messageUniqueId )
  {
  unsigned char *index;

  if ( messageUniqueId == 0 ) /* probably a timer */
    return;
  messageUniqueId -= 1;
  index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;
  index += ( unsigned char )( messageUniqueId / 8 );
  ( *index ) = ( *index ) & ~( 1 << messageUniqueId % 8 );
  }
#endif


/*
** FUNCTION RTDS_IsSemaphoreAvailable:
** -----------------------------------
** Checks if a semaphore is available
*/

#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
long RTDS_IsSemaphoreAvailable( RTDS_SemaphoreId semaphoreId )
  {
  return semaphoreId->isAvailable();
  }
#endif


/*
** FUNCTION RTDS_GetTimerUniqueId:
** -------------------------------
** Returns a new unique identifier for a timer.
*/

long RTDS_GetTimerUniqueId(void)
  {
  return 1;
  }


/*
** FUNCTION RTDS_StartTimer:
** -------------------------
** Starts a timer.
*/

void RTDS_StartTimer(RTDS_GlobalProcessInfo * context, long timerNumber, long timerUniqueId, int delay)
  {
  RTDS_TimerState * previousTimerDescriptor = NULL;
  RTDS_TimerState * timerDescriptor = NULL;
  RTDS_TimerState * startedTimerDescriptor = NULL;
  int               timerDelay = 0;
  
  /* We'll access the global list of timers: disable interrupts */
  RTDS_DISABLE_INTERRUPTS;
  /* Look for element in timer list just before the first active one */
  for ( previousTimerDescriptor = RTDS_globalTimerListFirst; previousTimerDescriptor != NULL; previousTimerDescriptor = previousTimerDescriptor->next )
    {
    if ( previousTimerDescriptor->next == RTDS_globalTimerListCurrent )
      break;
    }
  /* Look for the first timer in the list that is still active and that times-out after the one we start */
  for ( timerDescriptor = RTDS_globalTimerListCurrent; timerDescriptor != NULL; timerDescriptor = timerDescriptor->next )
    {
    /* If timer times out after the one we're starting, we've found its position */
    if ( timerDelay + timerDescriptor->timeoutDelay > delay )
      break;
    timerDelay += timerDescriptor->timeoutDelay;
    previousTimerDescriptor = timerDescriptor;
    }
  /* Insert timer at found position */
  startedTimerDescriptor = (RTDS_TimerState *)RTDS_MALLOC(sizeof(RTDS_TimerState));
  startedTimerDescriptor->timerNumber = timerNumber;
  startedTimerDescriptor->timerUniqueId = timerUniqueId;
  startedTimerDescriptor->timeoutValue = delay + RTDS_globalSystemTime;
  startedTimerDescriptor->timeoutDelay = delay - timerDelay;
  startedTimerDescriptor->receiverId = context->mySdlInstanceId;
  startedTimerDescriptor->state = RTDS_TIMER_OK;
  startedTimerDescriptor->next = timerDescriptor;
  if ( previousTimerDescriptor == NULL )
    {
    RTDS_globalTimerListFirst = startedTimerDescriptor;
    RTDS_globalTimerListCurrent = startedTimerDescriptor;
    }
  else
    {
    if ( previousTimerDescriptor->next == RTDS_globalTimerListCurrent )
      RTDS_globalTimerListCurrent = startedTimerDescriptor;
    previousTimerDescriptor->next = startedTimerDescriptor;
    }
  /* Update relative delay for next timer if any */
  if ( timerDescriptor != NULL )
    timerDescriptor->timeoutDelay -= startedTimerDescriptor->timeoutDelay;
  /* Enable back interrupts */
  RTDS_ENABLE_INTERRUPTS;
  }


/*
** FUNCTION RTDS_StopTimer:
** ------------------------
** Cancels a timer.
*/

void RTDS_StopTimer(RTDS_GlobalProcessInfo * context, long timerNumber)
  {
  RTDS_TimerState * timerDescriptor = NULL;
  
  /* Look for timer in list that has this timer number and is for this instance */
  for ( timerDescriptor = RTDS_globalTimerListFirst; timerDescriptor != NULL; timerDescriptor = timerDescriptor->next )
    {
    if ( ( timerDescriptor->timerNumber == timerNumber ) &&
         ( timerDescriptor->receiverId == context->mySdlInstanceId ) &&
         ( timerDescriptor->state == RTDS_TIMER_OK ) )
      {
      /* If found, just indicate the timer as cancelled in its descriptor */
      timerDescriptor->state = RTDS_TIMER_CANCELLED;
      return;
      }
    }
  /* If not found, cancel of a non-active timer: no big deal... */
  }


/*
** FUNCTION RTDS_ProcessForget:
** ----------------------------
** Forgets all information about a running process. Called when a process dies.
** Parameters:
** - RTDS_currentContext: context for the process to forget.
*/

void RTDS_ProcessForget( RTDS_GlobalProcessInfo * RTDS_currentContext )
  {
  RTDS_GlobalProcessInfo  * processInfo = NULL;
  RTDS_GlobalProcessInfo  * previousProcessInfo = NULL;
  RTDS_TimerState         * timer = NULL;
  RTDS_MessageHeader      * message = NULL;
  RTDS_MessageHeader      * tmpMessage = NULL;
  RTDS_SdlInstanceId      * instance2Delete = NULL;

  /* Free the current message if any */
  if ( RTDS_currentContext->currentMessage != NULL )
      {
#if defined( RTDS_SIMULATOR )
      /* Release the message unique id back to the pool */
      RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
#endif
      /* Free memory */
      RTDS_FREE( RTDS_currentContext->currentMessage );
      }

  /* Clean the timer chained list to free memory and stop watchdogs */
  for ( timer = RTDS_currentContext->timerList ; timer != NULL ; timer = RTDS_currentContext->timerList )
      {
      /* RTDS_TimerDelete( timer ); */
      RTDS_currentContext->timerList = timer->next;
      RTDS_FREE( timer );
      }

  /* Clean the save queue: free messages and message unique ids in the read and write save queues */
  message = RTDS_currentContext->readSaveQueue;
  while( message != NULL )
      {
#if defined( RTDS_SIMULATOR )
      RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
#endif
      tmpMessage = message;
      message = message->next;
      RTDS_FREE( tmpMessage );
      }

  message = RTDS_currentContext->writeSaveQueue;
  while( message != NULL )
      {
#if defined( RTDS_SIMULATOR )
      RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
#endif
      tmpMessage = message;
      message = message->next;
      RTDS_FREE( tmpMessage );
      }

  /* Forget and free the process information in the global instance chained list */
  previousProcessInfo = NULL;
  for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
      {
      /* If instance was found */
      if ( processInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )
          {
          RTDS_SIMULATOR_TRACE( RTDS_processDied , processInfo , NULL , RTDS_currentContext );
          /* Remove instance descriptor from list */
          if ( previousProcessInfo == NULL )
              RTDS_globalProcessInfo = processInfo->next;
          else
              previousProcessInfo->next = processInfo->next;

          /* Free the process info block */
          RTDS_FREE( processInfo->mySdlInstanceId );
          RTDS_FREE( processInfo );
          break;
          }
      previousProcessInfo = processInfo;
      }
  }


/*
** FUNCTION RTDS_TimerCleanUp:
** ---------------------------
** Called when the last received message is a timer to figure out if
** timer was not cancelled before it was received.
** Parameters:
** - currentContext: context of the current task
*/

void RTDS_TimerCleanUp(RTDS_GlobalProcessInfo * currentContext)
  {
  RTDS_TimerState * timer = NULL;
  RTDS_TimerState * prevTimer = NULL;

  prevTimer = NULL;
  for ( timer = currentContext->timerList ; timer != NULL; timer = timer->next )
    {
    /* If timer found and cancelled */
    if ( timer->timerUniqueId == currentContext->currentMessage->timerUniqueId )
      {
      if ( timer->state == RTDS_TIMER_CANCELLED )
        {
        /* Discard current message */
        RTDS_FREE( currentContext->currentMessage );
        currentContext->currentMessage = NULL;
        }
      /* Remove it from list of timers */
      if ( prevTimer == NULL )
        currentContext->timerList = currentContext->timerList->next;
      else
        prevTimer->next = timer->next;
      RTDS_FREE( timer );
      break;
      }
    prevTimer = timer;
    }
  }


/*
** FUNCTION RTDS_SemaphoreRegister:
** --------------------------------
** Registers a newly created semaphore in the global semaphore chained list.
** Parameters:
**  - parentScheduler: Parent scheduler for the current instance.
**  - semaphoreNumber: Identifier for the semaphore name.
**  - semaphoreId: Identifier for the semaphore itself.
**  - creatorContext: Context for the creator instance.
*/

void RTDS_SemaphoreRegister(RTDS_Scheduler * parentScheduler, int semaphoreNumber, RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * creatorContext)
  {
  long                        semIsAvailable = -1;
  RTDS_GlobalSemaphoreInfo  * new_semaphore_info;

  /* Register semaphore pseudo-process in parent scheduler */
  parentScheduler->registerInstance(semaphoreId, RTDS_internal_SemaphoreProcess, creatorContext, NULL);
  /* Allocate and fill in a new semaphore info structure */
  new_semaphore_info = (RTDS_GlobalSemaphoreInfo *)RTDS_MALLOC(sizeof(RTDS_GlobalSemaphoreInfo));

  new_semaphore_info->semaphoreId = semaphoreId;
  new_semaphore_info->semaphoreNumber = semaphoreNumber;

  /* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
  RTDS_CRITICAL_SECTION_START;
  new_semaphore_info->next = RTDS_globalSemaphoreInfo;
  RTDS_globalSemaphoreInfo = new_semaphore_info;
  RTDS_CRITICAL_SECTION_STOP;

#if defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACER)
  semIsAvailable = RTDS_IsSemaphoreAvailable(semaphoreId);
#endif
  RTDS_SIMULATOR_TRACE(RTDS_semaphoreCreated, semaphoreId, semIsAvailable, creatorContext);
  }


/*
** FUNCTION RTDS_BinarySemaphoreCreate:
** ------------------------------------
** Creates a new binary semaphore.
** Parameters:
**  - parentScheduler: Parent scheduler for the current instance.
**  - initialState: If 0, semaphore is created empty.
**  Returns the created semaphore.
*/

RTDS_SemaphoreId RTDS_BinarySemaphoreCreate(RTDS_Scheduler * parentScheduler, int initialState)
  {
  RTDS_BinarySemaphoreProcess * new_binary_semaphore;
  
  new_binary_semaphore = new RTDS_BinarySemaphoreProcess(parentScheduler);
  new_binary_semaphore->is_available = initialState;
  
  return (RTDS_SemaphoreId)new_binary_semaphore;
  }


/*
** FUNCTION RTDS_CountingSemaphoreCreate:
** --------------------------------------
** Creates a new counting semaphore.
** Parameters:
**  - parentScheduler: Parent scheduler for the current instance.
**  - initialCount: Initial number of available slots in semaphore.
** Returns the created semaphore.
*/

RTDS_SemaphoreId RTDS_CountingSemaphoreCreate(RTDS_Scheduler * parentScheduler, int initialCount)
  {
  RTDS_CountingSemaphoreProcess * new_counting_semaphore;
  
  new_counting_semaphore = new RTDS_CountingSemaphoreProcess(parentScheduler);
  new_counting_semaphore->available_slots = initialCount;
  
  return (RTDS_SemaphoreId)new_counting_semaphore;
  }


/*
** FUNCTION RTDS_MutexSemaphoreCreate:
** -----------------------------------
** Creates a new mutex semaphore.
** Parameters:
**  - parentScheduler: Parent scheduler for the current instance.
** Returns the created semaphore.
*/

RTDS_SemaphoreId RTDS_MutexSemaphoreCreate(RTDS_Scheduler * parentScheduler)
  {
  return (RTDS_SemaphoreId)(new RTDS_MutexSemaphoreProcess(parentScheduler));
  }


/*
** FUNCTION RTDS_GetSemaphoreId:
** -----------------------------
** Returns an actual semaphore descriptor corresponding to a semaphore name.
** Parameters:
**  - semaphoreNumber: Numerical identifier for the semaphore name.
** Returns the instance descriptor for the found semaphore.
*/

RTDS_SdlInstanceId * RTDS_GetSemaphoreId(int semaphoreNumber)
  {
  RTDS_GlobalSemaphoreInfo  * semaphore_info;
  RTDS_SemaphoreId            found_semaphore = NULL;

  RTDS_CRITICAL_SECTION_START;
  for ( semaphore_info = RTDS_globalSemaphoreInfo ; semaphore_info != NULL ; semaphore_info = semaphore_info->next )
    {
    if ( semaphore_info->semaphoreNumber == semaphoreNumber )
      {
      found_semaphore = semaphore_info->semaphoreId;
      break;
      }
    }
  RTDS_CRITICAL_SECTION_STOP;

  if ( found_semaphore == NULL )
    {
    RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_SEMAPHORE_ID )
    }
  return found_semaphore->RTDS_currentContext->mySdlInstanceId;
  }

