#include <stdio.h>

#include <RTDS_TTCN.h>
#include "RTDS_MACRO.h"
#include "RTDS_TTCN_MACRO.h"
#include "RTDS_OS.h"
#include "RTDS_String.h"
#include <RTDS_BasicTypes.h>
#include <RTDS_Common.h>
#include <tci/tci.h>

/*
** The following are declarations that are only valid in a TTCN-only non-debug context. If SDL
** is generated with the TTCN, or if generation is done for debug, these declarations will be
** in RTDS_Start.
*/
#if !defined(RTDS_TTCN_WITH_SDL) && !defined(RTDS_SIMULATOR)
/* Pointers to global list of information on processes and semaphores */
RTDS_GlobalProcessInfo    * RTDS_globalProcessInfo = NULL;
RTDS_GlobalSemaphoreInfo  * RTDS_globalSemaphoreInfo = NULL;

/* Various declarations for RTDS feature and critical sections */
RTDS_COVERAGE_DECL;
RTDS_SYS_TIME_DECL;
RTDS_START_SYNCHRO_DECL;
RTDS_CRITICAL_SECTION_DECL;
RTDS_SOCKET_ACCESS_DECL;
RTDS_CRITICAL_TRACE_SECTION_DECL;
RTDS_DTRACE_ACKNOWLEDGE_DECL;
#endif


/*
** RTDS_TTCN_Init:
** ---------------
** Performs all necessary initializations in a TTCN system. Must be called
** first in any generated executable.
*/
void RTDS_TTCN_Init(void)
  {
  /* Perform necessary initializations for RTDS features and critical sections if they are not done at SDL level */
#if !defined(RTDS_TTCN_WITH_SDL) && !defined(RTDS_SIMULATOR)
  RTDS_SYS_TIME_INIT;
  RTDS_COVERAGE_INIT;
  RTDS_DTRACE_ACKNOWLEDGE_INIT;
  RTDS_SOCKET_ACCESS_INIT;
  RTDS_CRITICAL_SECTION_INIT;
  RTDS_START_SYNCHRO_INIT;
  RTDS_CRITICAL_TRACE_SECTION_INIT;
#endif
  /* TTCN-specific initializations */
  /* (to be continued...) */
  }
  
  
/*
** RTDS_Register_Template_in_Pool:
** -------------------------------
** Add template to template pool and return it.
*/
RTDS_TTCN_Template * RTDS_TTCN_RegisterTemplatePool( RTDS_TTCN_GlobalComponentInfo * currentComponentInfo , RTDS_TTCN_Template * templateToRegister)
  {
  RTDS_TTCN_Template_Pool * newPoolElement = (RTDS_TTCN_Template_Pool*)RTDS_MALLOC(sizeof(RTDS_TTCN_Template_Pool));
  newPoolElement->templateID = templateToRegister;
  newPoolElement->next = currentComponentInfo->templatePool;
  currentComponentInfo->templatePool = newPoolElement;
  return templateToRegister;
  }

/*
** RTDS_TTCN_UpdateFirstMessage:
** -------------------------------
** Update first message of every port in component.
*/    
void RTDS_TTCN_UpdateFirstMessage ( RTDS_TTCN_GlobalComponentInfo * currentComponentInfo )
  {
  RTDS_GlobalProcessInfo        * RTDS_currentContext;
  RTDS_TTCN_PortMappingInfo     * portMapping;

  RTDS_CRITICAL_SECTION_START;
  RTDS_currentContext = currentComponentInfo->RTDS_currentContext;
  for (portMapping = currentComponentInfo->portMappingInfo; portMapping != NULL; portMapping = portMapping->next)
    {
    if (portMapping->currentMessage == NULL)
      {
      if (portMapping->queueControlBlock->queue != NULL)
        {
        RTDS_CHECKED_PTR_MALLOC(portMapping->currentMessage, 1, RTDS_MessageHeader);
        RTDS_MSG_QUEUE_ID_READ(portMapping->queueControlBlock, portMapping->currentMessage);
        }
      }
    }
  RTDS_CRITICAL_SECTION_STOP;
  }
    
/*
** RTDs_TTCN_UpdateTimerLists:
** -------------------------------
** Update timers lists for timeout.
*/ 
void RTDS_TTCN_UpdateTimerLists ( RTDS_TTCN_GlobalComponentInfo * RTDS_TTCN_currentComponent )
  {
  RTDS_TTCN_TimerInfoList * timerInfoList;
  RTDS_TIMER timerInfo;
  for (timerInfoList = RTDS_TTCN_currentComponent->timerInfoList ; timerInfoList!=NULL ; timerInfoList = timerInfoList->next)
    {
    timerInfo = timerInfoList->timerInfo;
    if (timerInfo->timerStatus == runningT)
        {
        if (timerInfo->timeoutValue < RTDS_GET_SYSTEM_TIME)
            {
            timerInfo->timerStatus = expiredT;
            }
        }
    }
  }
    
/*
** RTDS_SetStatus:
** ---------------
** Set component status.
*/
void RTDS_TTCN_SetStatus(RTDS_SdlInstanceId * componentToChange, ComponentStatusType status)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;

  /* Look for component to change in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If current component is component to change, set its status */
    if (componentInfo->RTDS_currentContext->mySdlInstanceId == componentToChange)
      {
      componentInfo->componentStatus = status;
      return;
      }
    }
  /* If component not found, error */
  RTDS_SYSTEM_ERROR(RTDS_ERROR_COMPONENT_NOT_FOUND);
  }

/*
** RTDS_CheckAlive:
** ----------------
** Check if a component is alive.
*/
int RTDS_TTCN_CheckAlive(RTDS_SdlInstanceId * componentToCheck)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;

  /* Look for component to check in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If current component is component to check */
    if (componentInfo->RTDS_currentContext->mySdlInstanceId == componentToCheck)
      {
      /* If the verdict of this component is inactive, running or stopped, component is alive */
      if ((componentInfo->componentStatus == inactiveC) || (componentInfo->componentStatus == runningC) || (componentInfo->componentStatus == stoppedC))
        return 1;
      }
    }
    /* If component not found, or verdict is not good, component is inactive */
    return 0;
  }

/*
** RTDS_CheckRunning:
** ------------------
** Check if a component is running.
*/    
int RTDS_TTCN_CheckRunning(RTDS_SdlInstanceId * componentToCheck)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;

  /* Look for component to check in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If current component is component to check */
    if (componentInfo->RTDS_currentContext->mySdlInstanceId == componentToCheck)
      {
      /* Checking if component is running via its status */
      if (componentInfo->componentStatus == runningC)
        return 1;
      }
    }
    /* If component not found, or status is not good, component is inactive */
    return 0;
  }

/*
** RTDS_CheckStopped:
** ------------------
** Check if a component is stopped.
*/   
int RTDS_TTCN_CheckStopped(RTDS_SdlInstanceId * componentToCheck)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;
  
  /* Look for component to check in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If current component is component to check, check if it's stopped via its status */
	if (componentInfo->RTDS_currentContext->mySdlInstanceId == componentToCheck)
      {
	  // return 1 if component is killed or stopped.
	  return ((componentInfo->componentStatus == stoppedC) || (componentInfo->componentStatus == killedC));
	  }
    }
    /* If component not found, assume it is stopped */
	return 1;
  }

/*
** RTDS_CheckKilled:
** -----------------
** Check if a component is killed.
*/  
int RTDS_TTCN_CheckKilled(RTDS_SdlInstanceId * componentToCheck)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;

  /* Look for component to check in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If current component is component to check, check if it's killed via its status */
    if (componentInfo->RTDS_currentContext->mySdlInstanceId == componentToCheck)
      {
      return (componentInfo->componentStatus == killedC);
      }
    }
    /* If component not found, assume it is stopped */
    return 1;
  }

/*
** RTDS_GetVerdict:
** ----------------
** Return the verdict for a component.
*/  
verdicttype RTDS_TTCN_GetVerdict(RTDS_SdlInstanceId * component)
  {
  RTDS_TTCN_GlobalComponentInfo * componentInfo;

  /* Look for component in global component list */
  for (componentInfo = RTDS_TTCN_globalComponentInfo; componentInfo != NULL; componentInfo = componentInfo->next)
    {
    /* If component found, return its verdict */
    if (componentInfo->RTDS_currentContext->mySdlInstanceId == component)
      return componentInfo->componentVerdict;   
    }
  /* If component not found, error */
  RTDS_SYSTEM_ERROR(RTDS_ERROR_COMPONENT_NOT_FOUND);
  }
    

/*
** RTDS_TTCN_GetFinalVerdict:
** ----------------
** Return the final verdict of a testcase.
*/  
verdicttype RTDS_TTCN_GetFinalVerdict()
  {
  verdicttype finalVerdict = TCI_VERDICT_NONE;
  /* For every components is component list */
  for (RTDS_TTCN_GlobalComponentInfo * RTDS_tmpglobalComponentInfo = RTDS_TTCN_globalComponentInfo; RTDS_tmpglobalComponentInfo!=NULL;RTDS_tmpglobalComponentInfo=  RTDS_tmpglobalComponentInfo->next)
    {
    /* If local verdict is worth, global verdict is local verdict */
    if (finalVerdict<RTDS_tmpglobalComponentInfo->componentVerdict)
      {
      finalVerdict=RTDS_tmpglobalComponentInfo->componentVerdict;
      }
    }
  return finalVerdict;
  }

/*
** RTDS_TTCN_getcurrentTestcase:
** ----------------
** get the name of the current running testcase.
*/  
char * RTDS_TTCN_getcurrentTestcase()
  {
  /* For every components is component list */
  for (RTDS_TTCN_GlobalComponentInfo * tmpGlobalComponentInfo = RTDS_TTCN_globalComponentInfo; tmpGlobalComponentInfo!=NULL;tmpGlobalComponentInfo=tmpGlobalComponentInfo->next)
    {
    if (tmpGlobalComponentInfo->componentKindType == TCI_MTC_COMP)
      {
	  return tmpGlobalComponentInfo->testcaseName;
      }
    }
  return "";
  }
    
/*
** RTDS_TTCN_MsgQueueCheck:
** ------------------------
** Check if there is any message in the queue associated with a port in a given component.
** If there is, the first message is put in the 'message' parameter, which must be allocated
** by the caller, and the function returns true. If there isn't, the function returns false.
*/     
int RTDS_TTCN_MsgQueueCheck(char * portName, RTDS_MessageHeader * message, RTDS_TTCN_GlobalComponentInfo * currentComponent)
  {
  RTDS_TTCN_PortMappingInfo * portMapping;

  /* Look for port in the list associated to the current component */
  for (portMapping = currentComponent->portMappingInfo; portMapping != NULL; portMapping = portMapping->next)
    {
    /* If port is the one to check */
    if (strcmp(portMapping->portId->portName, portName) == 0)
      {
      /* If queue is empty, over */
      if ( portMapping->queueControlBlock->queue == NULL )
        {
        return 0;
        }
      /* If queue is not empty */
      else
        {
        /* Copy first message in the queue */
        RTDS_SemTake(portMapping->queueControlBlock->chainedListSemId, RTDS_SEMAPHORE_TIME_OUT_FOREVER);
        memcpy(message, portMapping->queueControlBlock->queue, sizeof(RTDS_MessageHeader));
        RTDS_SemGive(portMapping->queueControlBlock->chainedListSemId);
        /* Over */
        return 1;
        }
      }
    }
  /* If port not found, assume no message found */
  return 0;
  }

/*
** RTDS_CheckAnyTimerRunning:
** --------------------------
** Check if a least one timer is running in a given component.
*/     
int RTDS_TTCN_CheckAnyTimerRunning (RTDS_TTCN_GlobalComponentInfo * currentComponent)
  {
  RTDS_TTCN_TimerInfoList * timerInfo;

  /* Browse timer list in component */
  for (timerInfo = currentComponent->timerInfoList; timerInfo !=NULL; timerInfo = timerInfo->next)
    {
    /* If timer is running, we have one */
    if (timerInfo->timerInfo->timerStatus == runningT)
      return 1;
    }
  /* If we get there, no timer found or none is running */
  return 0;
  }

/*
** RTDS_TTCN_CheckTimeout:
** ----------------------
** Check if timer is timeout in a given context.
*/     
int RTDS_TTCN_CheckTimeout(RTDS_TIMER timerToWaitFor)
  {
  if (timerToWaitFor->timerStatus == expiredT)
    {
    timerToWaitFor->timerStatus == inactiveT;
    return 0;
    }
  else
    {
    return 1;
    }
  }

/*
** RTDS_TTCN_TimerCreate:
** ----------------------
** Declare new timer with all necessary data.
*/  
RTDS_TIMER RTDS_TTCN_TimerCreate (long timerID, int timerNumber, TriTimerDuration duration)
  {
  RTDS_TIMER newTimer;
  RTDS_CHECKED_PTR_MALLOC(newTimer,1,RTDS_TTCN_TimerInfo);
  newTimer->timerID = timerID;
  #if defined( RTDS_SIMULATOR )
  newTimer->timerUniqueID = RTDS_GetMessageUniqueId();
  #endif
  newTimer->timerNumber = timerNumber;
  newTimer->duration = duration;
  newTimer->timerStatus=inactiveT;
  return newTimer;
  }
  
/*
** FUNCTION RTDS_TTCN_testcaseTerminated:
** --------------------------------------
** Function called when a testcase terminates to trace its final verdict
*/

void RTDS_TTCN_TestcaseTerminated(RTDS_GlobalProcessInfo * RTDS_currentContext, verdicttype verdict)
  {
  static char * verdict_strings[] = {"none", "pass", "inconc", "fail", "error"};
  char message[64];
  
  sprintf(message, "Testcase terminated: %s", verdict_strings[(int)verdict]);
  RTDS_INFORMATION(message);
  }    

  
RTDS_TTCN_ActivateAlstepList * RTDS_TTCN_activateAltstep(char * altstepName, RTDS_TTCN_functionpointer altstepfunction, RTDS_TTCN_GlobalComponentInfo * RTDS_TTCN_currentComponent)
  {
  RTDS_TTCN_ActivateAlstepList * RTDS_altstepToActivate = NULL;
  RTDS_CHECKED_PTR_MALLOC(RTDS_altstepToActivate,1,RTDS_TTCN_ActivateAlstepList);
  RTDS_altstepToActivate->altstepfunction = altstepfunction;
  RTDS_altstepToActivate->altstepName = altstepName;
  RTDS_altstepToActivate->next = RTDS_TTCN_currentComponent->activateAltstepList;
  RTDS_TTCN_currentComponent->activateAltstepList = RTDS_altstepToActivate;
  return RTDS_altstepToActivate;
  }