/* Copyright 2013 Humboldt University of Berlin
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>.
*
* Author: Mihal Brumbulli <mbrumbulli@gmail.com>
*/

#include <cstdlib>
#include <cstring>
#include <cmath>
#include <climits>
#include <cfloat>

#include "demoddix.h"

bool                    Demoddix::liveMode          = false;

int                     Demoddix::simulatorStatus_;
int                     Demoddix::displayStatus_;
CRITICAL_SECTION        Demoddix::statusLock_;
HANDLE                  Demoddix::simulatorThread;
std::string             Demoddix::simulatorExe;

unsigned int            Demoddix::bufferSize        = 1024;
FILE                  * Demoddix::traceFile         = NULL;
fpos_t                  Demoddix::tracePos;

std::vector<State>      Demoddix::stateList;
std::vector<Process>    Demoddix::processList;
std::vector<Semaphore>  Demoddix::semaphoreList;
std::vector<Message>    Demoddix::messageList;
std::vector<Node>       Demoddix::nodeList;
std::vector<Component>  Demoddix::componentList;
std::list<Packet>       Demoddix::packetList;

std::deque<Event>       Demoddix::eventList;	
unsigned long           Demoddix::currentEvent      = 0; 

long long               Demoddix::currentTime       = 0;
long long               Demoddix::beginTime         = 0;
long long               Demoddix::endTime           = 0;
long long               Demoddix::stepTime          = 1000000;


// Format for reading values from trace files
static const char     * stateF            = " <state id=\"s%u\" name=\"%[^\"]\" />";
static const char     * messageF          = " <message id=\"m%u\" name=\"%[^\"]\" />";
static const char     * semaphoreF        = " <semaphore id=\"x%u\" name=\"%[^\"]\" />";
static const char     * processF          = " <process id=\"p%u\" name=\"%[^\"]\" />";
static const char     * nodeF             = " <node id=\"n%u\" name=\"%[^\"]\" x=\"%lf\" y=\"%lf\" z=\"%lf\" />";
static const char     * componentF        = " <component id=\"c%u\" node=\"n%u\" name=\"%[^\"]\" />";
static const char     * traceF            = " <%*s nId=\"n%u\" cId=\"c%u\" time=\"%I64d\"";
static const char     * nodeChangedStateF = " <nodeChangedState nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" stateName=\"s%u\" prevStateName=\"s%u\" />";
static const char     * packetSentF       = " <packetSent nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" nReceiver=\"n%u\" pktName=\"m%u\" pktId=\"%u\" />";
static const char     * packetReceivedF   = " <packetReceived nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" nSender=\"n%u\" pktName=\"m%u\" pktId=\"%u\" />";


int Demoddix::simulatorStatus()
  { 
  EnterCriticalSection(&Demoddix::statusLock_);
  int value = Demoddix::simulatorStatus_;
  LeaveCriticalSection(&Demoddix::statusLock_);
  return value;
  }

void Demoddix::simulatorStatus(int value)
  { 
  EnterCriticalSection(&Demoddix::statusLock_);
  Demoddix::simulatorStatus_ = value;
  LeaveCriticalSection(&Demoddix::statusLock_);
  }

int Demoddix::displayStatus()
  {
  EnterCriticalSection(&Demoddix::statusLock_);
  int value = Demoddix::displayStatus_;
  LeaveCriticalSection(&Demoddix::statusLock_);
  return value;
  }

void Demoddix::displayStatus(int value)
  { 
  EnterCriticalSection(&Demoddix::statusLock_);
  Demoddix::displayStatus_ = value;
  LeaveCriticalSection(&Demoddix::statusLock_);
  }

void Demoddix::Open()
  {
  char buffer[Demoddix::bufferSize];
  
  double x, y, z;
  static double xMin = DBL_MAX, yMin = DBL_MAX, xMax = DBL_MIN, yMax = DBL_MIN;
  unsigned int id;
  char name[256];
  
  unsigned int nId, cId, nSender, nReceiver, pktId;
  unsigned int stateName, prevStateName, pktName;
  
  int simStat;
  
  fsetpos(Demoddix::traceFile, &Demoddix::tracePos);
  
  // There is smth to read
  while (fgets(buffer, Demoddix::bufferSize, Demoddix::traceFile) != NULL)
    {
    // Trace line is inclomplete, done
    if (buffer[strlen(buffer) - 1] != '\n')
      {
      simStat = Demoddix::simulatorStatus();
      if (simStat == Demoddix::OK || simStat == Demoddix::KO)
        {
        Demoddix::displayStatus(Demoddix::KO);
        }
      return;
      }
    // State
    if (sscanf(buffer, stateF, &id, name) == 2) 
      {
      if (Demoddix::stateList.size() < id + 1) 
        {
        Demoddix::stateList.resize(id + 1);
        }
      Demoddix::stateList[id] = State(std::string(name), rand() % 104, 1);
      }
    // Message
    else if (sscanf(buffer, messageF, &id, name) == 2) 
      {
      if (Demoddix::messageList.size() < id + 1) 
        {
        Demoddix::messageList.resize(id + 1);
        }
      Demoddix::messageList[id] = Message(std::string(name), rand() % 104);
      }
    // Process
    else if (sscanf(buffer, processF, &id, name) == 2) 
      {
      if (Demoddix::processList.size() < id + 1) 
        {
        Demoddix::processList.resize(id + 1);
        }
      Demoddix::processList[id] = Process(std::string(name));
      }
    // Semaphore
    else if (sscanf(buffer, semaphoreF, &id, name) == 2) 
      {
      if (Demoddix::semaphoreList.size() < id + 1) 
        {
        Demoddix::semaphoreList.resize(id + 1);
        }
      Demoddix::semaphoreList[id] = Semaphore(std::string(name));
      }
    // Node
    else if (sscanf(buffer, nodeF, &id, name, &x, &y, &z) == 5)
      {
      if (Demoddix::nodeList.size() < id + 1) 
        {
        Demoddix::nodeList.resize(id + 1);
        }
      Demoddix::nodeList[id] = Node(std::string(name), x, y, z, 0);
      xMin = std::min(x, xMin);
      yMin = std::min(y, yMin);
      xMax = std::max(x, xMax);
      yMax = std::max(y, yMax);
      }
    // Component
    else if (sscanf(buffer, componentF, &id, &nId, name) == 3)
      {
      if (Demoddix::componentList.size() < id + 1) 
        {
        Demoddix::componentList.resize(id + 1);
        }
      Demoddix::componentList[id] = Component(std::string(name), nId);
      }
    // State change
    else if (sscanf(buffer, nodeChangedStateF, &nId, &cId, &Demoddix::currentTime, &stateName, &prevStateName) == 5) 
      {
      Demoddix::eventList.push_back(Event(Demoddix::currentTime, nId, cId, Demoddix::tracePos));
      int oldPrio = Demoddix::stateList[Demoddix::nodeList[nId].state].priority;
      int newPrio = Demoddix::stateList[stateName].priority;
      if (newPrio > 0 && newPrio >= oldPrio) 
        {
        Demoddix::nodeList[nId].state = stateName;
        }
      if (Demoddix::liveMode)
        {
        fgetpos(Demoddix::traceFile, &Demoddix::tracePos);        
        return;
        }
      }
    // Sent
    else if (sscanf(buffer, packetSentF, &nSender, &cId, &Demoddix::currentTime, &nReceiver, &pktName, &pktId) == 6) 
      {
      Demoddix::eventList.push_back(Event(Demoddix::currentTime, nSender, cId, Demoddix::tracePos));
      Demoddix::packetList.push_back(Packet(nSender, nReceiver, pktName, pktId, Demoddix::eventList.back()));
      Demoddix::messageList[pktName].active = true;
      if (Demoddix::liveMode)
        {
        fgetpos(Demoddix::traceFile, &Demoddix::tracePos);        
        return;
        }
      }
    // Receive
    else if (sscanf(buffer, packetReceivedF, &nReceiver, &cId, &Demoddix::currentTime, &nSender, &pktName, &pktId) == 6) 
      {
      Demoddix::eventList.push_back(Event(Demoddix::currentTime, nReceiver, cId, Demoddix::tracePos));
      for (std::list<Packet>::iterator p = Demoddix::packetList.begin(); p != Demoddix::packetList.end(); ++p)
        {
        if (p->id == pktId)
          {
          Demoddix::packetList.erase(p);
          break;
          }
        }
      if (Demoddix::liveMode)
        {
        fgetpos(Demoddix::traceFile, &Demoddix::tracePos);        
        return;
        }
      }
    // Other event
    else if (Tracer::Send(buffer)) 
      {
      sscanf(buffer, traceF, &nId, &cId, &Demoddix::currentTime);
      Demoddix::eventList.push_back(Event(Demoddix::currentTime, nId, cId, Demoddix::tracePos));
      }
    // Everything else
    else
      {
      // Done loading
      if (std::string(buffer).find("<events>") != std::string::npos)
        {
        Demoddix::displayStatus(Demoddix::RUN);
        // Transform coordinates to fit screen
        double r = 2.0 / Window::MAIN_W; // Ratio for mapping pixels to coordinates
        double p = 4.0 * Window::POINT * r; // Padding
        for (unsigned long i = 0; i < Demoddix::nodeList.size(); ++i) 
          {
          Node &n = Demoddix::nodeList[i];
          n.x = 2.0 * (1.0 - p) * (n.x - xMin) / (xMax - xMin) - (1.0 - p);
          n.y = 2.0 * (1.0 - p) * (n.y - yMin) / (yMax - yMin) - (1.0 - p);
          }
        // Initialize tracers
        Tracer::Open();
        fgetpos(Demoddix::traceFile, &Demoddix::tracePos);
        return;
        }
      // Done tracing
      else if (std::string(buffer).find("</events>") != std::string::npos)
        {
        Demoddix::displayStatus(Demoddix::OK);
        // Initialize times
        if (!Demoddix::eventList.empty()) 
          {
          Demoddix::currentEvent = Demoddix::eventList.size() - 1;
          Demoddix::beginTime = Demoddix::eventList.front().time;
          Demoddix::endTime = Demoddix::eventList.back().time;
          Demoddix::currentTime = Demoddix::endTime;
          }
        // Setup lost packets
        for (std::list<Packet>::iterator p = Demoddix::packetList.begin(); p != Demoddix::packetList.end(); ++p)
          {
          p->event.lost = true;
          }
        return;
        }
      }
    // Trace line is complete, update position
    fgetpos(Demoddix::traceFile, &Demoddix::tracePos);
    }
  // Smth wrong with the trace
  simStat = Demoddix::simulatorStatus();
  if (simStat == Demoddix::OK || simStat == Demoddix::KO)
    {
    Demoddix::displayStatus(Demoddix::KO);
    }
  }

// Close trace file
void Demoddix::Close() 
  {
  if (Demoddix::traceFile != NULL)
    {
    fclose(Demoddix::traceFile);
    }
  if (Demoddix::simulatorStatus() != Demoddix::INIT)
    {
    if (Demoddix::simulatorStatus() == Demoddix::RUN)
      {
      // Force termination of simulation thread
      char sysPath[Demoddix::bufferSize];
      GetSystemDirectory(sysPath, Demoddix::bufferSize);
      std::string killCmd = sysPath;
      killCmd += "\\taskkill.exe /f /im ";
      size_t pos = Demoddix::simulatorExe.find_last_of('\\');
      if (pos != std::string::npos)
        {
        killCmd += Demoddix::simulatorExe.substr(pos + 1);
        }
      else
        {
        killCmd += Demoddix::simulatorExe;
        }
      system(killCmd.c_str());
      }
    WaitForSingleObject(Demoddix::simulatorThread, INFINITE);
    CloseHandle(Demoddix::simulatorThread);
    }
  // Destroy status mutex
  DeleteCriticalSection(&Demoddix::statusLock_);
  }

// Handle events forward until time is reached
void Demoddix::Forward() 
  {
  if (Demoddix::displayStatus() != Demoddix::OK)
    {
    return;
    }
  // Update current time
  Demoddix::currentTime += Demoddix::stepTime;
  if (Demoddix::currentTime > Demoddix::endTime) 
    {
    Demoddix::currentTime = Demoddix::endTime;
    }
  // Handle events
  char buffer[Demoddix::bufferSize];
  while (Demoddix::currentEvent < Demoddix::eventList.size()) 
    {
    Event &e = Demoddix::eventList[Demoddix::currentEvent];
    if (e.time > Demoddix::currentTime) 
      {
      break;
      }
    fsetpos(Demoddix::traceFile, &e.pos);
    if (fgets(buffer, Demoddix::bufferSize, Demoddix::traceFile) != NULL) 
      {
      Demoddix::Front(e, buffer);
      }
    ++Demoddix::currentEvent;
    }
  }

// Handle events backward until time is reached
void Demoddix::Rewind() 
  {
  if (Demoddix::displayStatus() != Demoddix::OK)
    {
    return;
    }
  // Update current time; if reached beginning then reset and return
  if (Demoddix::currentTime <= Demoddix::beginTime + Demoddix::stepTime) 
    {
    Demoddix::Reset();
    return;
    }
  Demoddix::currentTime -= Demoddix::stepTime;
  // Handle events
  char buffer[Demoddix::bufferSize];
  while (Demoddix::currentEvent > 0) 
    {
    --Demoddix::currentEvent;
    Event &e = Demoddix::eventList[Demoddix::currentEvent];
    if (e.time <= Demoddix::currentTime) 
      {
      ++Demoddix::currentEvent;
      break;
      }
		fsetpos(Demoddix::traceFile, &e.pos);
    if (fgets(buffer, Demoddix::bufferSize, Demoddix::traceFile) != NULL) 
      {
      Demoddix::Back(e, buffer);
      }
    }
  }

// Handle next event
void Demoddix::Next() 
  {
  if (Demoddix::displayStatus() != Demoddix::OK)
    {
    return;
    }
  // Handle event and update current time
  char buffer[Demoddix::bufferSize];
  while (Demoddix::currentEvent < Demoddix::eventList.size()) 
    {
    Event &e = Demoddix::eventList[Demoddix::currentEvent];
    Demoddix::currentTime = e.time;
    ++Demoddix::currentEvent;
    fsetpos(Demoddix::traceFile, &e.pos);
    if (fgets(buffer, sizeof(buffer), Demoddix::traceFile) != NULL) 
      {
      if (Demoddix::Front(e, buffer)) 
        {
        break;
        }
      }
    }
  }

// Handle previous event
void Demoddix::Previous()
  {
  if (Demoddix::displayStatus() != Demoddix::OK)
    {
    return;
    }
  // Handle event and update current time
  char buffer[Demoddix::bufferSize];
  while (Demoddix::currentEvent > 0) 
    {
    --Demoddix::currentEvent;
    Event &e = Demoddix::eventList[Demoddix::currentEvent];
    Demoddix::currentTime = e.time;
    fsetpos(Demoddix::traceFile, &e.pos);
    if (fgets(buffer, sizeof(buffer), Demoddix::traceFile) != NULL) 
      {
      if (Demoddix::Back(e, buffer)) 
        {
        break;
        }
      }
    }
  }

// Reset
void Demoddix::Reset() 
  {
  if (Demoddix::displayStatus() != Demoddix::OK)
    {
    return;
    }
  Demoddix::currentEvent = 0;
  Demoddix::packetList.clear();
  for (unsigned long i = 0; i < Demoddix::nodeList.size(); ++i)
    {
    Demoddix::nodeList[i].state = 0;
    }
  Demoddix::currentTime = Demoddix::beginTime;
  }

// Handle one event forward 
bool Demoddix::Front(Event &e, const char *buffer) 
  {
  unsigned int nId, cId, nSender, nReceiver, pktId;
  unsigned int stateName, prevStateName, pktName;
  long long time;
  // Handle state change by changing node color (if priority criteria are met) 
  if (sscanf(buffer, nodeChangedStateF, &nId, &cId, &time, &stateName, &prevStateName) == 5) 
    {
    int oldPrio = Demoddix::stateList[Demoddix::nodeList[nId].state].priority;
    int newPrio = Demoddix::stateList[stateName].priority;
    if (newPrio > 0 && newPrio >= oldPrio) 
      {
      Demoddix::nodeList[nId].state = stateName;
      }
    return true;
    }
  // Handle sent by appending a new packet to the list
  else if (sscanf(buffer, packetSentF, &nSender, &cId, &time, &nReceiver, &pktName, &pktId) == 6) 
    {
    Demoddix::packetList.push_back(Packet(nSender, nReceiver, pktName, pktId, e));
    return true;
    }
	// Handle receive by removing the first packet satisfying the criteria
  else if (sscanf(buffer, packetReceivedF, &nReceiver, &cId, &time, &nSender, &pktName, &pktId) == 6) 
    {
    for (std::list<Packet>::iterator p = Demoddix::packetList.begin(); p != Demoddix::packetList.end(); ++p)
      {
      if (p->id == pktId)
        {
        Demoddix::packetList.erase(p);
        break;
        }
      }
    return true;
    }
	// Handle other events via the tracer
  else 
    {
    if (Tracer::tracerList[e.nid].status() == Tracer::CONNECTED)
      {
      Tracer::Send(buffer);
      }
    return false;
    }
  }

// Handle one event backward
bool Demoddix::Back(Event &e, const char *buffer) 
  {	
  unsigned int nId, cId, nSender, nReceiver, pktId;
  unsigned int stateName, prevStateName, pktName;
  long long time;
  // Handle state change by changing node color (if priority criteria are met)
  if (sscanf(buffer, nodeChangedStateF, &nId, &cId, &time, &stateName, &prevStateName) == 5) 
    {
    int oldPrio = Demoddix::stateList[Demoddix::nodeList[nId].state].priority;
    int newPrio = Demoddix::stateList[prevStateName].priority;
    if (newPrio > 0 && newPrio >= oldPrio) 
      {
      Demoddix::nodeList[nId].state = prevStateName;
      }
    return true;
    }
  // Handle sent by removing the first packet satisfying the criteria
  else if (sscanf(buffer, packetSentF, &nSender, &cId, &time, &nReceiver, &pktName, &pktId) == 6) 
    {
    for (std::list<Packet>::reverse_iterator p = Demoddix::packetList.rbegin(); p != Demoddix::packetList.rend(); ++p)
      {
      if (p->id == pktId)
        {
        Demoddix::packetList.erase(--(p.base()));
        break;
        }
      }
    return true;
    }
  // Handle receive by appending a new packet to the list
  else if (sscanf(buffer, packetReceivedF, &nReceiver, &cId, &time, &nSender, &pktName, &pktId) == 6) 
    {
    Demoddix::packetList.push_back(Packet(nSender, nReceiver, pktName, pktId, e));
    return true;
    }
  else 
    {
    return false;
    }
  }

// Run simulation (if any)
DWORD WINAPI Demoddix::RunSimulation(void *arg)
  {
  short *runSim = (short *) arg;
  // No simulation to run, i.e., we are in trace mode
  if (*runSim == 0)
    {
    return 0;
    }
  // Run simulation, i.e., we are in run mode
  Demoddix::simulatorStatus(Demoddix::RUN);
  int r = system(Demoddix::simulatorExe.c_str());
  if (r != 0)
    {
    Demoddix::simulatorStatus(Demoddix::KO);
    return 0;
    }
  Demoddix::simulatorStatus(Demoddix::OK);
  return 0;
  }

// Main
int main(int argc, char **argv) 
  {
  if (argc != 3)
    {
    std::cerr << "Usage:" << std::endl << 
      "- " << argv[0] << " --run[-live] <executable>" << std::endl <<
      "- " << argv[0] << " --trace[-live] <trace-file>" << std::endl;
    return -1;
    }
  std::string optType = argv[1];
  if (optType.compare("--run") != 0 && optType.compare("--trace") != 0 && optType.compare("--run-live") != 0 && optType.compare("--trace-live") != 0)
    {
    std::cerr << "Usage:" << std::endl << 
      "- " << argv[0] << " --run[-live] <executable>" << std::endl <<
      "- " << argv[0] << " --trace[-live] <trace-file>" << std::endl;
    return -1;
    }
  // Check for live mode
  if (optType.find("-live") != std::string::npos)
    {
    Demoddix::liveMode = true;
    }
  // Initialize mutexes
  InitializeCriticalSection(&Demoddix::statusLock_);
  InitializeCriticalSection(&Tracer::pollLock_);
  // Trace file
  std::string optValue = argv[2];
  // Simulator thread parameters
  DWORD threadID;
  short runSim = 0;
  // Live
  if (optType.find("--run") != std::string::npos)
    {
    // Set status
    Demoddix::simulatorStatus(Demoddix::INIT);
    Demoddix::displayStatus(Demoddix::INIT);
    // Set executable
    Demoddix::simulatorExe = std::string(argv[2]);
    // Run simulation in its thread
    runSim = 1;
    Demoddix::simulatorThread = CreateThread(NULL, 0, Demoddix::RunSimulation, (void *) &runSim, 0, &threadID);
    // Delete old trace file if any
    remove((optValue + ".xml").c_str());
    // Wait until trace file is available
    while (Demoddix::traceFile == NULL)
      {
      int simStat = Demoddix::simulatorStatus();
      if (simStat == Demoddix::OK || simStat == Demoddix::KO)
        {
        break;
        }
      Sleep(500);
      Demoddix::traceFile = fopen((optValue + ".xml").c_str(), "rb");
      }
    if (Demoddix::traceFile != NULL)
      {
      fgetpos(Demoddix::traceFile, &Demoddix::tracePos);
      }
    else
      {
      Demoddix::displayStatus(Demoddix::KO);
      }
    }
  // Post-mortem
  else
    {
    // Set status
    Demoddix::simulatorStatus(Demoddix::OK);
    Demoddix::displayStatus(Demoddix::INIT);
    // Dummy thread
    Demoddix::simulatorThread = CreateThread(NULL, 0, Demoddix::RunSimulation, (void *) &runSim, 0, &threadID);
    // Sleep for some time
    Sleep(500);
    // Open trace
    Demoddix::traceFile = fopen(optValue.c_str(), "rb");
    if (Demoddix::traceFile != NULL)
      {
      fgetpos(Demoddix::traceFile, &Demoddix::tracePos);
      }
    else
      {
      Demoddix::displayStatus(Demoddix::KO);
      }
    }
  
  // Initialize GLUT
  argc = 1;
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGBA);
  
  // Create window
  RootWindow::Create();
  
  // GLUT loop
  glutSetOption(GLUT_ACTION_ON_WINDOW_CLOSE, GLUT_ACTION_CONTINUE_EXECUTION);
  glutMainLoop();
  
  // Terminate
  Tracer::Close();
  Demoddix::Close();
  
  return 0;
  }
