/* Copyright 2013 Humboldt University of Berlin
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>.
*
* Author: Mihal Brumbulli <mbrumbulli@gmail.com>
*/

#include <cstring>
#include <climits>
#include <cstdlib>
#include <winsock2.h>

#include "demoddix.h"


std::vector<Tracer>     Tracer::tracerList;
std::vector<HANDLE>     Tracer::threadList;

HANDLE                  Tracer::pollThread;
std::string             Tracer::tracerCommand;

bool                    Tracer::doPoll_         = false;
CRITICAL_SECTION        Tracer::pollLock_;
unsigned int            Tracer::pollInterval    = 200;

// Format for reading values from trace files
static const char       * taskCreatedF      = " <taskCreated nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" creatorId=\"%u\" pName=\"p%u\" creatorName=\"p%u\" pId=\"%u\" />";
static const char       * taskDeletedF      = " <taskDeleted nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" pId=\"%u\" />";
static const char       * messageSentF      = " <messageSent nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" mId=\"%u\" pId=\"%u\" sigNum=\"%u\" msgName=\"m%u\" />";
static const char       * messageReceivedF  = " <messageReceived nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" mId=\"%u\" pId=\"%u\" sigNum=\"%u\" msgName=\"m%u\" />";
static const char       * messageSavedF     = " <messageSaved nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" mId=\"%u\" pId=\"%u\" sigNum=\"%u\" msgName=\"m%u\" />";
static const char       * semaphoreCreatedF = " <semaphoreCreated nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" semName=\"x%u\" stillAvailable=\"%d\" pId=\"%u\" />";
static const char       * takeAttemptF      = " <takeAttempt nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" semName=\"x%u\" timeout=\"%d\" pId=\"%u\" semId=\"%u\" />";
static const char       * takeSuccededF     = " <takeSucceeded nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" semName=\"x%u\" stillAvailable=\"%d\" pId=\"%u\" semId=\"%u\" />";
static const char       * takeTimedOutF     = " <takeTimedOut nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" semName=\"x%u\" pId=\"%u\" semId=\"%u\" />";
static const char       * giveSemF          = " <giveSem nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" semName=\"x%u\" pId=\"%u\" semId=\"%u\" />";
static const char       * timerStartedF     = " <timerStarted nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" timerName=\"m%u\" pId=\"%u\" tId=\"%u\" timeLeft=\"%d\" />";
static const char       * timerCancelledF   = " <timerCancelled nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" timerName=\"m%u\" pId=\"%u\" tId=\"%u\" />";
static const char       * timerTimedOutF    = " <timerTimedOut nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" timerName=\"m%u\" pId=\"%u\" tId=\"%u\" />";
static const char       * taskChangedStateF = " <taskChangedState nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" pId=\"%u\" stateName=\"s%u\" />";
static const char       * informationF      = " <information nId=\"n%u\" cId=\"c%u\" time=\"%I64d\" pName=\"p%u\" pId=\"%u\" message=\"%[^\"]\" />";

// Constructor
Tracer::Tracer()
  {
  InitializeCriticalSection(&lock_);
  EnterCriticalSection(&lock_);
  port_ = 0;
  sock_ = INVALID_SOCKET;
  status_ = IDLE; 
  LeaveCriticalSection(&lock_);
  }

// Destructor
Tracer::~Tracer()
  {
  DeleteCriticalSection(&lock_);
  }

// Atomic getters and setters
unsigned short Tracer::port()
  { 
  EnterCriticalSection(&lock_);
  unsigned short value = port_;
  LeaveCriticalSection(&lock_);
  return value;
  }

void Tracer::port(unsigned short value)
  { 
  EnterCriticalSection(&lock_);
  port_ = value;
  LeaveCriticalSection(&lock_);
  }

SOCKET Tracer::sock()
  { 
  EnterCriticalSection(&lock_);
  SOCKET value = sock_;
  LeaveCriticalSection(&lock_);
  return value;
  }

void Tracer::sock(SOCKET value)
  { 
  EnterCriticalSection(&lock_);
  sock_ = value;
  LeaveCriticalSection(&lock_);
  }

int Tracer::status()
  { 
  EnterCriticalSection(&lock_);
  int value = status_;
  LeaveCriticalSection(&lock_);
  return value;
  }

void Tracer::status(int value)
  { 
  EnterCriticalSection(&lock_);
  status_ = value;
  LeaveCriticalSection(&lock_);
  }

// Atomic polling
bool Tracer::doPoll() 
  {
  EnterCriticalSection(&Tracer::pollLock_);
  bool value = Tracer::doPoll_;
  LeaveCriticalSection(&Tracer::pollLock_);
  return value;
  }

void Tracer::doPoll(bool value)
  { 
  EnterCriticalSection(&Tracer::pollLock_);
  Tracer::doPoll_ = value;
  LeaveCriticalSection(&Tracer::pollLock_);
  }

// Initialize
void Tracer::Open() 
  {
  // Initialize sockets
  WSADATA wsaData;
  WSAStartup(MAKEWORD(2, 2), &wsaData);
  // Look for tracer using environment variables
  char *tracer = getenv("PRAGMADEV_TRACER_COMMAND");
  if (tracer != NULL)
    {
    Tracer::tracerCommand = std::string(tracer) + " --nmw";
    }
  else
    {
    tracer = getenv("RTDS_HOME");
    if (tracer != NULL)
      {
      Tracer::tracerCommand = "runSAMscTracer-windows --nmw";
      }
    }
  Tracer::tracerList.resize(Demoddix::nodeList.size());
  Tracer::threadList.resize(Demoddix::nodeList.size());
  Tracer::doPoll(true);
  DWORD threadID;
  Tracer::pollThread = CreateThread(NULL, 0, &Tracer::Poll, NULL, 0, &threadID);
  }

// Terminate
void Tracer::Close() 
  {
  // If not initialized, done
  if (!Tracer::doPoll())
    {
    DeleteCriticalSection(&Tracer::pollLock_);
    return;
    }
  // Terminate polling thread
  Tracer::doPoll(false);
  WaitForSingleObject(Tracer::pollThread, INFINITE);
  CloseHandle(Tracer::pollThread);
  DeleteCriticalSection(&Tracer::pollLock_);
  // Terminate tracer threads
  for (unsigned long i = 0; i < Tracer::tracerList.size(); ++i)
    {
    if (Tracer::tracerList[i].status() != Tracer::IDLE)
      {
      WaitForSingleObject(Tracer::threadList[i], INFINITE);
      CloseHandle(Tracer::threadList[i]);
      }
    }
  // Cleanup sockets
  WSACleanup();
  }

// Launch
void Tracer::Launch(unsigned long id) 
  {
  // No tracer, done
  if (Tracer::tracerCommand.empty())
    {
    return;
    }
  // Make sure trace is not already running
  if (Tracer::tracerList[id].status() != Tracer::IDLE) 
    {
    return;
    }
  // Remember used ports from running tracers
  std::vector<unsigned short> used;
  for (unsigned long i = 0; i < Tracer::tracerList.size(); ++i) 
    {
    Tracer &t = Tracer::tracerList[i];
    if (t.port() > 0) 
      {
      used.push_back(t.port());
      }
    }
  // Find a free port
  unsigned short p = USHRT_MAX;
  for (; p > 0; --p) 
    {
    bool exists = false;
    for (unsigned long i = 0; i < used.size(); ++i)
      {
      if (used[i] == p)
        {
        exists = true;
        break;
        }
      }
    if (exists) 
      {
      continue;
      }
    // Create socket
    SOCKET sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (sock == INVALID_SOCKET)
      {
      continue;
      }
    // Make non-blocking
    unsigned long argp = 1;
    if (ioctlsocket(sock, FIONBIO, &argp) == SOCKET_ERROR)
      {
      closesocket(sock);
      continue;
      }
    // Make socket reusable
    int yes = 1;
    if (setsockopt(sock, SOL_SOCKET, SO_EXCLUSIVEADDRUSE, (const char*) &yes, sizeof(yes)) == SOCKET_ERROR)
      {
      closesocket(sock);
      continue;
      }
    // Setup address info
    struct sockaddr_in addr;
    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(p);
    // Bind
    if (bind(sock, (struct sockaddr *) &addr, sizeof(addr)) == SOCKET_ERROR)
      {
      closesocket(sock);
      continue;
      }
    closesocket(sock);
    break;
    }
	// No free port
  if (p == 0) 
    {
    std::cerr << "Warning: no free port for tracer on node " << id << "." << std::endl;
    return;
    }
	// Create thread for tracer
  Tracer::tracerList[id].port(p);
  unsigned int *tid = (unsigned int *) malloc(sizeof(unsigned int));
  *tid = id;
  DWORD threadID;
  Tracer::threadList[id] = CreateThread(NULL, 0, &Tracer::Create, (void *) tid, 0, &threadID);
  }

DWORD WINAPI Tracer::Create(void *arg)
  {
  unsigned int *id = (unsigned int *) arg;
  Tracer::tracerList[*id].status(Tracer::OPENED);
  char cmd[256];
  sprintf(cmd, "%s -p %d -n Node[%d]", Tracer::tracerCommand.c_str(), Tracer::tracerList[*id].port(), *id);
  system(cmd);
  Tracer::tracerList[*id].status(Tracer::CLOSED);
  free(id);
  return 0;
  }

// Continously check tracer status
DWORD WINAPI Tracer::Poll(void *arg)
  {
  while (Tracer::doPoll()) 
    {
    for (unsigned long id = 0; id < Tracer::tracerList.size(); ++id) 
      {
      switch (Tracer::tracerList[id].status())
        {
        // If tracer is idle, do nothing
        case Tracer::IDLE:
        continue;
        // If tracer is opened, try to connect
        case Tracer::OPENED:
        // Create socket if not done already
        if (Tracer::tracerList[id].sock() == INVALID_SOCKET)
          {
          Tracer::tracerList[id].sock(socket(AF_INET, SOCK_STREAM, IPPROTO_TCP));
          if (Tracer::tracerList[id].sock() == INVALID_SOCKET)
            {
            continue;
            }
          }
        // Setup address info
        struct sockaddr_in addr;
        memset(&addr, 0, sizeof(addr));
        addr.sin_family = AF_INET;
        addr.sin_addr.s_addr = inet_addr("127.0.0.1");
        addr.sin_port = htons(Tracer::tracerList[id].port());
				// Connect
				if (connect(Tracer::tracerList[id].sock(), (struct sockaddr *) &addr, sizeof(addr)) != SOCKET_ERROR) 
          {
          Tracer::tracerList[id].status(Tracer::CONNECTED);
          }
        break;
        // If tracer is connected, do nothing
        case Tracer::CONNECTED:
        break;
        // If tracer is closed, terminate thread
        case Tracer::CLOSED:
        closesocket(Tracer::tracerList[id].sock());
        Tracer::tracerList[id].port(0);
        Tracer::tracerList[id].sock(INVALID_SOCKET);
        Tracer::tracerList[id].status(Tracer::IDLE);
        WaitForSingleObject(Tracer::threadList[id], INFINITE);
        CloseHandle(Tracer::threadList[id]);
        break;
        }
      }
    // Sleep for some time
    Sleep(Tracer::pollInterval);
    }
  return 0;
  }

// Format trace and send it to the tracer
bool Tracer::Send(const char* buffer)
  {
  bool isValid = true;
  char command[Demoddix::bufferSize];
  long long time;
  unsigned int nId, cId, creatorId, pId, semId, tId, mId;
  unsigned int pName, creatorName, sigNum, msgName, semName, timerName, stateName;
  int stillAvailable, timeout, timeLeft;
  char info[256];
  if (sscanf(buffer, taskCreatedF, &nId, &cId, &time, &creatorId, &pName, &creatorName, &pId) == 7) 
    {
		sprintf(command, "taskCreated| -t%I64d| -c0x%I64X| -n%s| -N%s| 0x%I64X|\n", 
      (time - Demoddix::beginTime) / 1000000, (((unsigned long long) cId) << 32) + creatorId, Demoddix::processList[pName].name.c_str(), Demoddix::processList[creatorName].name.c_str(), (((unsigned long long) cId) << 32) + pId);
    }
  else if (sscanf(buffer, taskDeletedF, &nId, &cId, &time, &pName, &pId) == 5) 
    {
    sprintf(command, "taskDeleted| -t%I64d| -n%s| 0x%I64X|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), (((unsigned long long) cId) << 32) + pId);
    }
  else if (sscanf(buffer, messageSentF, &nId, &cId, &time, &pName, &mId, &pId, &sigNum, &msgName) == 8) 
    {
    sprintf(command, "messageSent| -t%I64d| -n%s| -i%u| 0x%I64X| %u| %s|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), mId, (((unsigned long long) cId) << 32) + pId, sigNum, Demoddix::messageList[msgName].name.c_str());
    }
  else if (sscanf(buffer, messageReceivedF, &nId, &cId, &time, &pName, &mId, &pId, &sigNum, &msgName) == 8) 
    {
    sprintf(command, "messageReceived| -t%I64d| -n%s| -i%u| 0x%I64X| %u| %s|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), mId, (((unsigned long long) cId) << 32) + pId, sigNum, Demoddix::messageList[msgName].name.c_str());
    }
  else if (sscanf(buffer, messageSavedF, &nId, &cId, &time, &pName, &mId, &pId, &sigNum, &msgName) == 8) 
    {
    sprintf(command, "messageSaved| -t%I64d| -n%s| -i%u| 0x%I64X| %u| %s|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), mId, (((unsigned long long) cId) << 32) + pId, sigNum, Demoddix::messageList[msgName].name.c_str());
    }
  else if (sscanf(buffer, semaphoreCreatedF, &nId, &cId, &time, &semName, &stillAvailable, &pId) == 6) 
    {
    sprintf(command, "semaphoreCreated| -t%I64d| -s%s| -a%d| 0x%I64X|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::semaphoreList[semName].name.c_str(), stillAvailable, (((unsigned long long) cId) << 32) + pId);
    }
  else if (sscanf(buffer, takeAttemptF, &nId, &cId, &time, &pName, &semName, &timeout, &pId, &semId) == 8) 
    {
    sprintf(command, "takeAttempt| -t%I64d| -n%s| -s%s| -T%d| 0x%I64X| 0x%I64X|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::semaphoreList[semName].name.c_str(), timeout, (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + semId);
    }
  else if (sscanf(buffer, takeSuccededF, &nId, &cId, &time, &pName, &semName, &stillAvailable, &pId, &semId) == 8) 
    {
    sprintf(command, "takeSucceeded| -t%I64d| -n%s| -s%s| -a%d| 0x%I64X| 0x%I64X|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::semaphoreList[semName].name.c_str(), stillAvailable, (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + semId);
    }
  else if (sscanf(buffer, takeTimedOutF, &nId, &cId, &time, &pName, &semName, &pId, &semId) == 7) 
    {
    sprintf(command, "takeTimedOut| -t%I64d| -n%s| -s%s| 0x%I64X| 0x%I64X|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::semaphoreList[semName].name.c_str(), (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + semId);
    }
  else if (sscanf(buffer, giveSemF, &nId, &cId, &time, &pName, &semName, &pId, &semId) == 7) 
    {
    sprintf(command, "giveSem| -t%I64d| -n%s| -s%s| 0x%I64X| 0x%I64X|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::semaphoreList[semName].name.c_str(), (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + semId);
    }
  else if (sscanf(buffer, timerStartedF, &nId, &cId, &time, &pName, &timerName, &pId, &tId, &timeLeft) == 8) 
    {
    sprintf(command, "timerStarted| -t%I64d| -n%s| -T%s| 0x%I64X| 0x%I64X| %d|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::messageList[timerName].name.c_str(), (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + tId, timeLeft);
    }
  else if (sscanf(buffer, timerCancelledF, &nId, &cId, &time, &pName, &timerName, &pId, &tId) == 7) 
    {
    sprintf(command, "timerCancelled| -t%I64d| -n%s| -T%s| 0x%I64X| 0x%I64X|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::messageList[timerName].name.c_str(), (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + tId);
    }
  else if (sscanf(buffer, timerTimedOutF, &nId, &cId, &time, &pName, &timerName, &pId, &tId) == 7) 
    {
    sprintf(command, "timerTimedOut| -t%I64d| -n%s| -T%s| 0x%I64X| 0x%I64X|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::messageList[timerName].name.c_str(), (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + tId);
    }
  else if (sscanf(buffer, taskChangedStateF, &nId, &cId, &time, &pName, &pId, &stateName) == 6) 
    {
    sprintf(command, "taskChangedState| -t%I64d| -n%s| 0x%I64X| %s|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), (((unsigned long long) cId) << 32) + pId, Demoddix::stateList[stateName].name.c_str());
    }
  else if (sscanf(buffer, informationF, &nId, &cId, &time, &pName, &pId, info) == 6) 
    {
    sprintf(command, "information| -t%I64d| -n%s| 0x%I64X| %s|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), (((unsigned long long) cId) << 32) + pId, info);
    }
  else 
    {
    isValid = false;
    }
  // If trace is valid and tracer is connected, try to send via socket
  if (isValid && Tracer::tracerList[nId].status() == Tracer::CONNECTED)
    {
    // If send fails, close socket 
    if (send(Tracer::tracerList[nId].sock(), command, strlen(command), 0) == SOCKET_ERROR) 
      {
      Tracer::tracerList[nId].status(Tracer::IDLE);
      }
    }
  return isValid;
  }
