#include <stdio.h>
#include <stdlib.h>

#include "tci/tci.h"
#include "tci/tciRequired.h"
#include "RTDS_TTCN_MACRO.h"
#include "RTDS_TTCN.h"

/* Include unistd when using a unix based system or cygwin */
#ifdef __CYGWIN__
#include <unistd.h>
#else
#include <stdlib.h>
#endif

/*
** Declarations of functions provided by the generated code, but that are not
** in the TCI required interface.
*/
TciModuleIdListType * RTDS_TTCN_GetAllRootModules();
verdicttype RTDS_TTCN_ExecuteTestcase(char *);
int RTDS_TTCN_GetProcessNumber(TciModuleIdType *);

extern char * RTDS_TTCN_defaultRootModule;


#ifdef RTDS_TTCN_SUT_INIT
int RTDS_TTCN_SUT_INIT(void);
#endif

/* Maximum length for a name */
#define MAX_NAME 256


/*
** STATIC VARIABLES:
** -----------------
*/

static  int errorIndicator = 0;               /* Set to true if an error occurs */
static  int controlPartTerminated = 0;        /* Set to true when the currently run control part ends */
static  int globalVerdict = TCI_VERDICT_NONE; /* Global verdict for all excuted testcases */

/* String representations for verdicts */
static char * verdict_string[] = { "none", "pass", "inconc", "fail", "error" };

/* Type for linked list of strings, allowing to sort list of modules or testcases */
typedef struct _StringLinkedList
  {
  char                      * string;
  int                         index;
  struct _StringLinkedList  * next;
  } StringLinkedList;


/*
** FUNCTION sortedInsert:
** ----------------------
** Utility function to insert a string in a sorted string linked list.
*/

static StringLinkedList * sortedInsert(StringLinkedList * list, char * string)
  {
  StringLinkedList  * element;
  StringLinkedList  * previous;
  StringLinkedList  * current;
  
  element = (StringLinkedList*)malloc(sizeof(StringLinkedList));
  element->string = string;
  element->index = 0;
  element->next = NULL;
  if ( list == NULL )
    return element;
  previous = NULL; current = list;
  while ( current != NULL )
    {
    if ( strcmp(element->string, current->string) <= 0 )
      {
      element->next = current;
      if ( previous == NULL )
        list = element;
      else
        previous->next = element;
      break;
      }
    if ( current->next == NULL )
      {
      current->next = element;
      break;
      }
    previous = current;
    current = current->next;
    }

  return list;
  }


/*
** FUNCTION sortedInsert:
** ----------------------
** Utility function to free a whole string linked list.
*/

static void freeStringLinkedList(StringLinkedList * list)
  {
  StringLinkedList  * next;
  while ( list != NULL )
    {
    next = list->next;
    free(list);
    list = next;
    }
  }


/*
** FUNCTION tciError:
** -----------------------
** Part of the TCI provided interface. Called when an error occurs in the test.
*/

void tciError(char* message)
  {
  printf("!! Error!: %s\n" , message);
  }


/*
** FUNCTION tciControlTerminated:
** ------------------------------
** Part of the TCI provided interface. Called when the execution of a control part ends.
*/

void tciControlTerminated()
  {
  controlPartTerminated = 1;
  }


/*
** FUNCTION tciTestCaseStarted:
** ----------------------------
** Part of the TCI provided interface. Called when the execution of a testcase starts.
*/

void tciTestCaseStarted(TciTestCaseIdType * testCaseId, TciParameterListType * parameterList, double timer)
  {
  printf("-> Testcase %s started\n" , testCaseId->baseName);
  }


/*
** FUNCTION tciTestCaseTerminated:
** -------------------------------
** Part of the TCI provided interface. Called when the execution of a testcase ends.
*/

void tciTestCaseTerminated(VerdictValue verdict, TciParameterListType* parameterlist)
  {
  printf("<- Verdict is: %s\n", verdict_string[verdict]);
  if (globalVerdict < verdict)
    globalVerdict = verdict;
  }


/*
** FUNCTION executeControlPart:
** ----------------------------
** Executes the control part in the current root module if any.
*/

static int executeControlPart(void)
  {
  TriComponentId  * startedControlpart;
  verdicttype       finalVerdict;
  
  printf("=> Starting control part...\n");
  controlPartTerminated = 0;
  globalVerdict = TCI_VERDICT_NONE;
  startedControlpart = tciStartControl();
  if (startedControlpart == NULL)
    {
    printf("== Error while starting control part!\n");
    return TCI_VERDICT_ERROR;
    }
  for ( ; ; )
    {
    RTDS_SLEEP( 50 );
    if ( controlPartTerminated )
      break;
    }
  printf("<= Control part terminated\n");
  return (int)globalVerdict;
  }


/*
** FUNCTION executeTestcase:
** -------------------------
** Executes a testcase identified by its name in the current root module.
*/

static int executeTestcase(char * testcaseName)
  {
  return (int)RTDS_TTCN_ExecuteTestcase(testcaseName);
  }


/*
** FUNCTION main:
** ==============
** If called with a command in its arguments, executes this command in the default module,
** which is the one the code generation has been run on. If called without parameters,
** allows to run control parts or testcases in all known modules interactively.
*/

int main(int argc, char *argv[])
	{
  QualifiedName           currentModuleId;              /* TCI identifier for the current module */
  char                    currentModuleName[MAX_NAME];  /* Name for the current module */
	TciModuleIdListType   * moduleList;                   /* List of available modules */
  int                     moduleIndex;                  /* Index of a module in the previous list */
  StringLinkedList      * sortedNames;                  /* Sorted linked list of strings, i.e module or testcase names */
  StringLinkedList      * sortedNamesElement;           /* Element in the previous list */
  char                    moduleName[MAX_NAME];         /* Name for a module */
  QualifiedName           tempModuleId;                 /* Temporary TCI identifier for tests */
  short                   hasControl;                   /* Indicates whether a module has a control part or not */
  unsigned int            nbTestcases;                  /* Number of testcases in a module */
  TciTestCaseIdListType * testcasesList;                /* List of testcases in a module */
  int                     testcaseIndex;                /* Index of a testcase in the previous list */
  char                    testcaseName[MAX_NAME];       /* Name of a testcase */
  char                  * endPointer;                   /* End pointer for a conversion of a string to an integer */
  int                     finalVerdict;                 /* Final verdict in testcase or control part execution */

  /* Initializations for TTCN */
  RTDS_TTCN_Init();

  /* Start SDL if any */
#ifdef RTDS_TTCN_SUT_INIT
  RTDS_TTCN_SUT_INIT();
#endif

  /* Set default root module */
  currentModuleId.moduleName = currentModuleName;
  currentModuleId.baseName = currentModuleName;
  strcpy(currentModuleName, RTDS_TTCN_defaultRootModule);
  tciRootModule(&currentModuleId);
  
  /* If execution is not interactive */
  if (argc > 1)
    {
    /* Check the command on the command line is correct */
    if ( ( argc == 2 || argc == 3 ) && strcmp(argv[1], "start") == 0 )
      {
      /* Execute control part or testcase depending on command argument, then exit */
      if ( argc == 2 )
        return executeControlPart();
      else
        return executeTestcase(argv[2]);
      }
    /* If command is incorrect, show usage */
    else
      {
      printf("Usage: %s [ start [ <testcase name> ] ]\n", argv[0]);
      printf(" - With no parameters, will go in ineractive mode, allowing to execute any control part");
      printf("   or any testcase in any module.\n");
      printf(" - 'start' will start the control part for the default root module.\n");
      printf(" - 'start <testcase name>' will start the testcase with this name in the default module.\n");
      printf("Default root module is %s.\n", RTDS_TTCN_defaultRootModule);
      return 0;
      }
    }
  /* If execution is interactive, display banner */
  printf("********************************************************\n");
  printf("**                  P R A G M A D E V                 **\n");
  printf("**             Real Time Developer Studio             **\n");
  printf("** -------------------------------------------------- **\n");
  printf("** TTCN test control interactive command line utility **\n");
  printf("********************************************************\n");
  
  /* Interactive loop */
  for ( ; ; )
    {
    /* Display sorted list of available modules */
    printf("\nAvailable modules:\n");
    moduleList = RTDS_TTCN_GetAllRootModules();
    sortedNames = NULL;
    for (moduleIndex = 0; moduleIndex < moduleList->length; moduleIndex++)
      sortedNames = sortedInsert(sortedNames, moduleList->modList[moduleIndex].moduleName);
    moduleIndex = 1;
    sortedNamesElement = sortedNames;
    while ( sortedNamesElement != NULL )
      {
      /* Figure out if module has a control part and/or testcases */
      tempModuleId.moduleName = sortedNamesElement->string;
      tempModuleId.baseName = sortedNamesElement->string;
      hasControl = (RTDS_TTCN_GetProcessNumber(&tempModuleId) != 0);
      tciRootModule(&tempModuleId);
      testcasesList = tciGetTestCases();
      nbTestcases = 0;
      if ( testcasesList != NULL )
        {
        for ( testcaseIndex = 0; testcaseIndex < testcasesList->length; testcaseIndex++ )
          {
          if ( strcmp(testcasesList->testcaseList[testcaseIndex].moduleName, sortedNamesElement->string) == 0 )
            nbTestcases ++;
          }
        }
      /* If module has a control part or testcases, print it */
      if ( hasControl || nbTestcases != 0 )
        {
        sortedNamesElement->index = moduleIndex;
        printf("%4d ", moduleIndex);
        if ( strcmp(sortedNamesElement->string, currentModuleName) == 0 )
          printf("*");
        else
          printf(" ");
        printf(" %s (", sortedNamesElement->string);
        if ( hasControl )
          {
          printf("control part");
          if ( nbTestcases != 0 ) printf(" & ");
          }
        if ( nbTestcases != 0 )
          printf("%d testcase%s", nbTestcases, nbTestcases == 1 ? "" : "s");
        printf(")\n");
        moduleIndex ++;
        }
      sortedNamesElement = sortedNamesElement->next;
      }
    printf("\nCurrent module is marked with a '*'.\n");
    /* Set back root module, as it may have been modified above */
    tciRootModule(&currentModuleId);

    /* Input new current module */
    printf("\nEnter new current module.\n");
    printf("Number or name - <return> = no change - . or Ctrl-D = quit: ");
    if ( fgets(moduleName, MAX_NAME, stdin) == NULL )
      {
      printf("\n");
      return 0;
      }
    if ( moduleName[0] == '.' )
      return 0;
    /* If a new module was specified */
    if ( moduleName[0] != '\0' && moduleName[0] != '\n' )
      {
      /* Try to get a number from the input string */
      moduleIndex = strtol(moduleName, &endPointer, 10);
      /* If it worked, get actual module name */
      if ( endPointer != &(moduleName[0]) )
        {
        sortedNamesElement = sortedNames;
        while ( sortedNamesElement != NULL )
          {
          if ( sortedNamesElement->index == moduleIndex )
            {
            strcpy(moduleName, sortedNamesElement->string);
            break;
            }
          sortedNamesElement = sortedNamesElement->next;
          }
        if (sortedNamesElement == NULL)
          {
          printf("\nInvalid module number %d!\n", moduleIndex);
          printf("Press return to continue:");
          fgets(moduleName, MAX_NAME, stdin);
          continue;
          }
        }
      /* If module name was specified directly, make sure to remove the ending \n if any */
      else
        {
        int i = 0;
        while ( moduleName[i] != '\0' )
          {
          if ( moduleName[i] == '\n' )
            {
            moduleName[i] = '\0';
            break;
            }
          i++;
          }
        }
      
      /* Set new root module */
      strcpy(currentModuleName, moduleName);
      tciRootModule(&currentModuleId);
      }
    /* Sorted module names list is no more needed */
    freeStringLinkedList(sortedNames);

    /* Display testcases in the new current module */
    testcasesList = tciGetTestCases();
    if ( testcasesList == NULL )
      {
      printf("\nInvalid module name %s!\n", currentModuleName);
      printf("Press return to continue:");
      fgets(moduleName, MAX_NAME, stdin);
      continue;
      }
    printf("\n\nAvailable control part or testcases in module %s:\n", currentModuleName);
    sortedNames = NULL;
    for( testcaseIndex = 0; testcaseIndex < testcasesList->length; testcaseIndex++ )
      {
      if ( strcmp(testcasesList->testcaseList[testcaseIndex].moduleName, currentModuleName) == 0 )
        sortedNames = sortedInsert(sortedNames, testcasesList->testcaseList[testcaseIndex].baseName);
      }
    if ( RTDS_TTCN_GetProcessNumber(&currentModuleId) != 0 )
      printf("%5d   control\n", 0);
    testcaseIndex = 1;
    sortedNamesElement = sortedNames;
    while ( sortedNamesElement != NULL )
      {
      sortedNamesElement->index = testcaseIndex;
      printf("%5d   %s\n", testcaseIndex, sortedNamesElement->string);
      testcaseIndex++;
      sortedNamesElement = sortedNamesElement->next;
      }
    /* Input testcase to run */
    printf("\nEnter control part or testcase to run.\n");
    printf("Number or name - <return> = go back - . or Ctrl-D = quit: ");
    if ( fgets(testcaseName, MAX_NAME, stdin) == NULL )
      {
      printf("\n");
      return 0;
      }
    if ( testcaseName[0] == '.' )
      return 0;
    /* If no testcase was specified, go back */
    if ( testcaseName[0] == '\0' || testcaseName[0] == '\n')
      continue;
    /* Try to get a number from the input string */
    testcaseIndex = strtol(testcaseName, &endPointer, 10);
    /* If it worked, get actual testcase name */
    if ( endPointer != &(testcaseName[0]) )
      {
      if ( testcaseIndex == 0 )
        {
        strcpy(testcaseName, "control");
        }
      else
        {
        sortedNamesElement = sortedNames;
        while ( sortedNamesElement != NULL )
          {
          if ( sortedNamesElement->index == testcaseIndex )
            {
            strcpy(testcaseName, sortedNamesElement->string);
            break;
            }
          sortedNamesElement = sortedNamesElement->next;
          }
        if (sortedNamesElement == NULL)
          {
          printf("\nInvalid testcase number %d!\n", testcaseIndex);
          printf("Press return to continue:");
          fgets(testcaseName, MAX_NAME, stdin);
          continue;
          }
        }
      }
    /* If testcase name was specified directly, make sure to remove the ending \n if any */
    else
      {
      int i = 0;
      while ( testcaseName[i] != '\0' )
        {
        if ( testcaseName[i] == '\n' )
          {
          testcaseName[i] = '\0';
          break;
          }
        i++;
        }
      }
    /* Sorted testcase names list is no more needed */
    freeStringLinkedList(sortedNames);
    
    /* Run control part or specified testcase */
    printf("\n\n");
    if ( strcmp(testcaseName, "control") == 0 )
      {
      printf("***** CONTROL PART EXECUTION **************\n");
      finalVerdict = executeControlPart();
      }
    else
      {
      printf("***** TESTCASE EXECUTION ******************\n");
      finalVerdict = executeTestcase(testcaseName);
      }
    printf("Final verdict: %s\n" , verdict_string[finalVerdict]);
    printf("***** EXECUTION END ***********************\n\n");
    }
  }
